package com.iplatform.base.service;

import com.iplatform.base.util.CategoryUtils;
import com.iplatform.base.util.TextUtils;
import com.iplatform.model.po.S_category;
import com.iplatform.model.vo.CategoryTreeVo;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.jdbc.service.BaseServiceImpl;
import org.springframework.stereotype.Service;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 系统设置分类目录，电商系统重构使用。
 * <p>1.后续要加上缓存！</p>
 * @author 时克英
 * @date 2023-05-15
 */
@Service
public class CategoryServiceImpl extends BaseServiceImpl {

    private static final String SQL_MAX_ID = "select max(id) from s_category";

    /**
     * 更新分类信息，同时会更新父级、下级相关启用状态。
     * <pre>
     *     1)如状态为关闭，那么所以子集的状态都关闭
     *     2)如是开启，则父类的状态为开启
     * </pre>
     * @param category
     */
    public void execUpdateCategory(S_category category){
        this.save(category);
        if(category.getStatus().intValue() == 0){
            //如状态为关闭，那么所以子集的状态都关闭
            this.execute("update s_category set status=0 where path like ?", new Object[]{"/" + category.getId() + "/"});
        } else {
            //如是开启，则父类的状态为开启
            this.updatePidStatusById(category.getId());
        }
    }

    /**
     * 开启父级状态
     *
     * @param id Integer
     * @return Boolean
     */
    private Boolean updatePidStatusById(Integer id) {
        S_category category = this.get(new S_category(id));
        if(category == null){
            return true;
        }
        List<Integer> categoryIdList = TextUtils.stringToArrayByRegex(category.getPath(), StringUtils.FOLDER_SEPARATOR);
        categoryIdList.removeIf(i -> i.equals(0));
        if (StringUtils.isEmptyList(categoryIdList)) {
            return false;
        }
        Map<String, Object> param = new HashMap<>(2);
        param.put("ids", categoryIdList);
        this.execute("update s_category set status=1 where id in (:ids)", param);
        return true;
    }

    /**
     * 查询给定父ID，包含子类数量
     * @param pid
     * @return
     * @date 2023-05-17
     */
    public int queryChildCategorySize(int pid){
        return this.queryForInt("select count(id) from s_category where pid=?", new Object[]{pid});
    }

    /**
     * 获得下一个可用的最大ID。
     * @return
     * @date 2023-05-17
     */
    public int queryNextId(){
        int maxId = this.queryForInt(SQL_MAX_ID, new Object[]{});
        return maxId+1;
    }

    /**
     * 查找是否有相同的类型记录
     * @param name 分类名称
     * @param type 类别
     * @param owner 归属
     * @param id 已有ID
     * @return
     * @date 2023-05-17
     */
    public int queryNameUnique(String name, Integer type, Integer owner, int id){
        Map<String, Object> param = new HashMap<>(4);
        StringBuilder sql = new StringBuilder("select count(id) from s_category where owner=:owner and name=:name");
        param.put("owner", owner);
        param.put("name", name);

        if(type != null){
            sql.append(" and type=:type");
            param.put("type", type);
        }
        if(id > 0){
            sql.append(" and id<>:id");
            param.put("id", id);
        }
        return this.queryForInt(sql.toString(), param);
    }

    /**
     * 使用缓存对象替换数据库查询。
     * @param type
     * @param status
     * @param name
     * @param owner
     * @return
     * @date 2023-05-17
     */
    @Deprecated
    public List<CategoryTreeVo> getListTree(Integer type, Integer status, String name, int owner) {
//        name = URLUtil.decode(name);
        if(StringUtils.isNotEmpty(name)){
            try {
                name = URLDecoder.decode(name, StringUtils.DEFAULT_CHARSET_UTF8);
            } catch (UnsupportedEncodingException e) {
                throw new RuntimeException(e);
            }
        }
        return getTree(type, status, name, null, owner);
    }

    /**
     * 带结构的无限级分类.
     * @author 时克英
     * @date 2023-05-15
     */
    @Deprecated
    private List<CategoryTreeVo> getTree(Integer type, Integer status, String name, List<Integer> categoryIdList, int owner) {
        //循环数据，把数据对象变成带list结构的vo
        List<CategoryTreeVo> treeList = new ArrayList<>();

        Map<String, Object> parameters = new HashMap<>();
        StringBuilder sql = new StringBuilder(SQL_GET_TREE);

        parameters.put("owner", owner);

        if(!StringUtils.isEmptyList(categoryIdList)){
            sql.append(" and id in (:ids)");
            parameters.put("ids", categoryIdList);
        }
        if(type != null){
            sql.append(" and type = :type");
            parameters.put("type", type);
        }
        if(status != null && status.intValue() >= 0){
            sql.append(" and status = :status");
            parameters.put("status", status);
        }
        sql.append(" order by sort desc, id asc");

        List<S_category> allTree = this.select(sql.toString(), parameters, new S_category());
        if(StringUtils.isEmptyList(allTree)){
            return new ArrayList<>(2);
        }

        // 根据名称搜索特殊处理 这里仅仅处理两层搜索后有子父级关系的数据
//        if (StrUtil.isNotBlank(name) && CollUtil.isNotEmpty(allTree)) {
//            List<Category> searchCategory = new ArrayList<>();
//            List<Integer> categoryIds = allTree.stream().map(Category::getId).collect(Collectors.toList());
//
//            List<Integer> pidList = allTree.stream().filter(c -> c.getPid() > 0 && !categoryIds.contains(c.getPid()))
//                    .map(Category::getPid).distinct().collect(Collectors.toList());
//            if (CollUtil.isNotEmpty(pidList)) {
//                pidList.forEach(pid -> {
//                    searchCategory.add(dao.selectById(pid));
//                });
//            }
//            allTree.addAll(searchCategory);
//        }
        for (S_category category : allTree) {
            treeList.add(CategoryUtils.toCategoryTreeVo(category));
        }

        Map<Integer, CategoryTreeVo> map = new HashMap<>(treeList.size());
        //ID 为 key 存储到map 中
        for (CategoryTreeVo categoryTreeVo : treeList) {
            map.put(categoryTreeVo.getId(), categoryTreeVo);
        }

        List<CategoryTreeVo> list = new ArrayList<>();
        CategoryTreeVo parentTree = null;
        for (CategoryTreeVo tree : treeList) {
            //子集ID返回对象，有则添加。
            parentTree = map.get(tree.getPid());
            if (parentTree != null) {
                parentTree.getChild().add(tree);
            } else {
                list.add(tree);
            }
        }
//        System.out.println("无限极分类 : getTree:" + JSON.toJSONString(list));
        return list;
    }

    private static final String SQL_GET_TREE = "select * from s_category where owner=:owner";
}
