package com.iplatform.base.service;

import com.iplatform.model.po.S_dept;
import com.iplatform.model.po.S_dict_data;
import com.iplatform.model.po.S_dict_type;
import com.walker.db.page.GenericPager;
import com.walker.db.page.ListPageContext;
import com.walker.db.page.PageSearch;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.jdbc.service.BaseServiceImpl;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 数据字典表操作。
 * @date 2022-09-19
 */
@Service
public class CodeServiceImpl extends BaseServiceImpl {

    private static final String SQL_PAGE_TYPE_PREFIX = "select * from s_dict_type where 1=1";
    private static final String SQL_PAGE_DATA_PREFIX = "select * from s_dict_data where 1=1";

    private static final String SQL_QUERY_EXIST_DATA = "select * from s_dict_data where dict_type=? and dict_value=?";

    public void execDeleteDictData(Long[] dictCodes){
        if(dictCodes == null || dictCodes.length == 0){
            return;
        }
        List<Object[]> parameters = new ArrayList<>(8);
        for(long dictCode : dictCodes){
            Object[] p = new Object[]{dictCode};
            parameters.add(p);
        }
        this.execBatchUpdate("delete from s_dict_data where dict_code=?", parameters);
    }

    private static final String SQL_DICT_TREE_LIST = "select * from s_dict_data where dict_type=? order by parent_id, dict_sort";

    /**
     * 返回代码树结构，需要的列表集合。<p></p>
     * 为前端展示树结构准备数据，集合已经按照父节点顺序排好顺序。
     * @param dictType 代码类型，如：question_root_catalog
     * @return
     * @date 2023-03-13
     */
    public List queryDictTreeList(String dictType){
        return this.select(SQL_DICT_TREE_LIST, new Object[]{dictType}, new S_dict_data());
    }

    /**
     * 返回所有代码项集合。<p></p>
     * 方法用在代码缓存功能。
     * @return
     * @date 2023-03-10
     */
    public List<S_dict_data> queryAllCodeItemList(){
        return this.select("select * from s_dict_data order by dict_sort", new Object[]{}, new S_dict_data());
    }

    /**
     * 返回数据字典所有根集合，即：字典表集合。<p></p>
     * 该方法缓存使用。
     * @return
     * @date 2023-03-10
     */
    public List<S_dict_type> queryRootCodeList(){
        return this.selectAll(new S_dict_type());
    }

    public S_dict_data queryOneDictData(long dictCode){
        return this.get(new S_dict_data(dictCode));
    }

    /**
     * 字典表中是否存在给定值一样的字典项。并返回字典项。
     * @param dictType
     * @param value
     * @return
     * @date 2022-11-20
     */
    public S_dict_data queryOneDictData(String dictType, String value){
        return this.get(SQL_QUERY_EXIST_DATA, new Object[]{dictType, value}, new S_dict_data());
    }

    /**
     * 获取一个字典类型详情。
     * @param dictId 字典id
     * @return
     * @date 2022-11-19
     */
    public S_dict_type queryOneDictType(long dictId){
        return this.get("select * from s_dict_type where dict_id=?", new Object[]{dictId}, new S_dict_type());
    }

    /**
     * 分页返回字典类型(代码表)列表
     * @param dictName
     * @param dictType
     * @param status
     * @param startTime
     * @param endTime
     * @return
     * @date 2022-11-16
     */
    public GenericPager<S_dict_type> queryPageDictType(String dictName, String dictType, int status, long startTime, long endTime){
        Map<String, Object> parameters = new HashMap<>();
        StringBuilder sql = new StringBuilder(SQL_PAGE_TYPE_PREFIX);

        if(StringUtils.isNotEmpty(dictName)){
            sql.append(" and dict_name like :dictName");
            parameters.put("dictName", "%" + dictName + "%");
        }
        if(StringUtils.isNotEmpty(dictType)){
            sql.append(" and dict_type = :dictType");
            parameters.put("dictType", dictType);
        }
        if(status >= 0){
            sql.append(" and status = :status");
            parameters.put("status", status);
        }
        if(startTime > 0){
            sql.append(" and create_time >= :startTime");
            parameters.put("startTime", startTime);
        }
        if(endTime > 0){
            sql.append(" and create_time <= :endTime");
            parameters.put("endTime", endTime);
        }

        PageSearch pageSearch = ListPageContext.getPageSearch();
        return this.selectSplit(sql.toString(), parameters, pageSearch.getPageIndex(), pageSearch.getPageSize(), new S_dict_type());
    }

    /**
     * 分页查询数据字典项列表。
     * @param dictType 字典类型(名称)
     * @param dictLabel 模糊查询字典项名称
     * @return
     * @date 2022-11-19
     */
    public GenericPager<S_dict_data> queryPageDictData(String dictType, String dictLabel){
        Map<String, Object> parameters = new HashMap<>();
        StringBuilder sql = new StringBuilder(SQL_PAGE_DATA_PREFIX);
        if(StringUtils.isNotEmpty(dictType)){
            sql.append(" and dict_type = :dictType");
            parameters.put("dictType", dictType);
        }
        if(StringUtils.isNotEmpty(dictLabel)){
            sql.append(" and dict_label like :dictLabel");
            parameters.put("dictLabel", "%" + dictLabel + "%");
        }
        PageSearch pageSearch = ListPageContext.getPageSearch();
        return this.selectSplit(sql.toString(), parameters, pageSearch.getPageIndex(), pageSearch.getPageSize(), new S_dict_data());
    }

    /**
     * 根据字典类型名称(代码表)，返回子代码集合。
     * @param dictType 代码类型
     * @return
     */
    public List<S_dict_data> queryDictDataByType(String dictType){
        return this.select(new S_dict_data(), "where status=0 and dict_type=? order by dict_sort asc", new Object[]{dictType});
    }
}
