package com.iplatform.base.service;

import com.iplatform.model.po.S_dept;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.jdbc.service.BaseServiceImpl;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Service
public class DeptServiceImpl extends BaseServiceImpl {

    private static final String WHERE_QUERY_ROOT_LIST = "where parent_id=0 order by order_num";
    private static final String WHERE_QUERY_ROOT_LIST_BY_ID = "where parent_id=0 and id=? order by order_num";
    private static final String WHERE_QUERY_ROOT_ONE_CHILDREN = "where org_id=:orgId";
    private static final String QUERY_SUB_SIZE_IN_CURRENT = "select count(id) total from s_dept where parent_id=?";
    private static final String QUERY_USER_SIZE_IN_CURRENT = "select count(id) total from s_user_core where dept_id=?";
    private static final String QUERY_DEPT_FOR_CACHE = "select * from s_dept order by parent_id, order_num";
    private static final String QUERY_TREE_ALL = "select * from s_dept where del_flag=0 and status=0 order by parent_id, order_num";
    private static final String QUERY_TREE_ONE = "select * from s_dept where org_id=? and del_flag=0 and status=0 order by parent_id, order_num";

    /**
     * 返回所有根机构列表，也就是第一层独立单位(集团)
     * @param orgRootId 给定的单位ID，如果提供该参数，则表明只返回一条记录
     * @return
     * @date 2022-12-01
     */
    public List<S_dept> queryRootOrgList(long orgRootId){
        if(orgRootId <= 0){
            return this.select(new S_dept(),WHERE_QUERY_ROOT_LIST, new Object[]{});
        } else {
            return this.select(new S_dept(),WHERE_QUERY_ROOT_LIST_BY_ID, new Object[]{orgRootId});
        }
    }

    /**
     * 机构管理显示机构列表，由前端组织树结果。<p></p>
     * 注意: 这个列表包含根机构，也就是一个完整的独立机构列表，如果没有根节点，前端无法选择添加。2022-12-02
     * @param orgRootId 根机构ID，必填
     * @param deptName 查询的机构名称
     * @return
     * @date 2022-12-01
     */
    public List<S_dept> queryRootOrgChildrenList(long orgRootId, String deptName){
        Map<String, Object> parameters = new HashMap<>();
        parameters.put("orgId", orgRootId);
        StringBuilder sql = new StringBuilder(WHERE_QUERY_ROOT_ONE_CHILDREN);
        if(StringUtils.isNotEmpty(deptName)){
            sql.append(" and dept_name like :deptName");
            parameters.put("deptName", "%" + deptName + "%");
        }
        sql.append(" order by parent_id, order_num");
        return this.select(new S_dept(), sql.toString(), parameters);
    }

    public S_dept queryOneDept(long id){
        return this.get(new S_dept(id));
    }

    /**
     * 查询给定单位下存在多少个子机构。
     * @param id 当前机构ID
     * @return 返回子机构数量
     */
    public int querySubDeptSizeInCurrent(long id){
        return this.sqlMathQuery(QUERY_SUB_SIZE_IN_CURRENT, new Object[]{id}, Integer.class);
    }

    /**
     * 查询给定机构下存在多少用户
     * @param id 当前机构ID
     * @return 返回用户数量
     * @date 2022-12-03
     */
    public long queryUserSizeInCurrent(long id){
        return this.sqlMathQuery(QUERY_USER_SIZE_IN_CURRENT, new Object[]{id}, Long.class);
    }

    /**
     * 缓存使用该方法，加载所有机构列表。
     * @return
     * @date 2022-12-03
     */
    public List<S_dept> queryAllDeptListForCache(){
        return this.select(QUERY_DEPT_FOR_CACHE, new Object[]{}, new S_dept());
    }

    /**
     * 获取机构列表，为前端树管理展示用。
     * @param orgId 指定的顶级单位ID，如果没指定则展示所有顶级机构。
     * @return
     * @date 2022-12-08
     */
    public List<S_dept> queryOrgListForTree(long orgId){
        if(orgId <= 0){
            // 不存在指定的单位，就展示全部顶级点位
            return this.select(QUERY_TREE_ALL, new Object[]{}, new S_dept());
        } else {
            // 只显示给定顶级机构下的列表
            return this.select(QUERY_TREE_ONE, new Object[]{orgId}, new S_dept());
        }
    }
}
