package com.iplatform.base.service;

import com.iplatform.model.po.S_menu;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.jdbc.service.BaseServiceImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * 菜单权限数据操作。
 * @author 时克英
 * @date 2022-11-02
 */
@Service
public class MenuServiceImpl extends BaseServiceImpl {

    protected final transient Logger logger = LoggerFactory.getLogger(getClass());

    private static final String SQL_QUERY_ROLE_MENUS = "select DISTINCT(menu_id) from s_role_menu where role_id in (:roleIdList)";

    private static final String SQL_SAME_MENU_PARENT = "select * from s_menu where parent_id=? and menu_name = ?";

    private static final String SQL_MENU_USER = "select count(menu_id) total from s_role_menu where menu_id=?";

    /**
     * 查询给定菜单是否被分配到角色功能中
     * @param menuId
     * @return 存在返回大于0
     * @date 2022-12-29
     */
    public int queryRoleMenuSize(String menuId){
        return this.sqlMathQuery(SQL_MENU_USER, new Object[]{menuId}, Integer.class);
    }

    /**
     * 查询给定父节点下是否存在已重名的菜单。
     * @param parentId
     * @param menuName
     * @return
     * @date 2022-12-26
     */
    public S_menu queryExistMenuInParent(String parentId, String menuName){
        List<S_menu> list = this.select(SQL_SAME_MENU_PARENT, new Object[]{parentId, menuName}, new S_menu());
        if(StringUtils.isEmptyList(list)){
            return null;
        }
        return list.get(0);
    }

    /**
     * 给定角色ID集合，返回对应菜单ID列表
     * @param roleIdList
     * @return
     * @date 2022-11-12
     */
    public List<String> queryRoleMenuIdList(List<String> roleIdList){
        if(StringUtils.isEmptyList(roleIdList)){
            return null;
        }
        // 2023-07-12 安全模块传递的角色id，可能存在字符串，如：ROLE_USER等，应当过滤掉，只保留系统角色ID（数值）
        //-------------------------------------------------
        List<Long> roleIdValueList = new ArrayList<>(4);
        for(String roleId: roleIdList){
            if(StringUtils.isNumeric(roleId)){
                roleIdValueList.add(Long.parseLong(roleId));
            }
        }
        if(StringUtils.isEmptyList(roleIdValueList)){
            return null;
        }
        //-------------------------------------------------
//        Map<String, Object> param = new HashMap<>(2);
//        param.put("roleIdList", roleIdValueList);
//        List<Map<String, Object>> list = this.select(SQL_QUERY_ROLE_MENUS, param);
        // 因为 oracle报错：where role_id in ([2])，所以更换写法！2023-07-19
        MapSqlParameterSource sqlParameterSource = new MapSqlParameterSource();
        sqlParameterSource.addValue("roleIdList", roleIdValueList);
        List<Map<String, Object>> list = this.queryListObjectWhereIn(SQL_QUERY_ROLE_MENUS, sqlParameterSource);
        logger.debug("查询角色权限集合: {}", list);
        if(list != null && list.size() > 0){
            List<String> menuIdList = new ArrayList<>(64);
            for(Map<String, Object> map : list){
//                menuIdList.add(map.get("menu_id").toString());
                // 2023-07-19 由于oracle返回的是否大写：MENU_ID，因此这里可以取第一个字段（因为只有一个）
                // 而不能通过小写取menu_id，否则取不到数据。
                menuIdList.add(map.values().toArray()[0].toString());
            }
            return menuIdList;
        }
        return null;
    }

    /**
     * 查询系统所有角色与菜单对应关系集合。在security权限拦截中使用该方法
     * @return 集合中包括：role_id, menu_id
     * @date 2022-11-02
     */
    public List<Map<String, Object>> queryRolesPermList(){
        return this.select("select * from s_role_menu", new Object[]{});
    }
}
