package com.iplatform.base.service;

import com.iplatform.model.po.S_role;
import com.iplatform.model.po.S_user_core;
import com.walker.db.page.GenericPager;
import com.walker.db.page.ListPageContext;
import com.walker.db.page.PageSearch;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.jdbc.service.BaseServiceImpl;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Service
public class RoleServiceImpl extends BaseServiceImpl {

    private static final String SQL_ORG_ROLES = "select * from s_role where org_id=? and del_flag=0 and status=0";

    private static final String SQL_USER_ROLES = "select r.* from s_role_user ru, s_role r where ru.user_id=? and ru.role_id=r.role_id";

    private static final String SQL_PAGE_ROLE_PREFIX = "select * from s_role where org_id=:orgId";

    private static final String SQL_UPDATE_ROLE_STATUS = "update s_role set status=? where role_id=?";

    private static final String SQL_DEL_ROLE_MENU = "delete from s_role_menu where role_id=?";
    private static final String SQL_DEL_ROLE_USER = "delete from s_role_user where role_id=?";
    private static final String SQL_DEL_ROLE_AND_USER = "delete from s_role_user where role_id=? and user_id=?";
    private static final String SQL_INSERT_ROLE_USER = "insert into s_role_user(user_id, role_id, org_id) values(?,?,0)";

    /**
     * 批量添加角色对应的用户。
     * @param roleId
     * @param userIdList
     * @date 2022-12-22
     */
    public void execInsertRoleUserList(long roleId, Long[] userIdList){
        List<Object[]> parameters = new ArrayList<>();
        Object[] one = null;
        for(long userId : userIdList){
            one = new Object[2];
            one[0] = userId;
            one[1] = roleId;
            parameters.add(one);
        }
        this.execBatchUpdate(SQL_INSERT_ROLE_USER, parameters);
    }

    /**
     * 删除一个角色中的一个用户。
     * @param roleId
     * @param userId
     * @date 2022-12-21
     */
    public void execDeleteRoleUser(long roleId, long userId){
        this.execute(SQL_DEL_ROLE_AND_USER, new Object[]{roleId, userId});
    }

    public void execDeleteRole(long roleId){
        this.execute(SQL_DEL_ROLE_MENU, new Object[]{roleId});
        this.execute(SQL_DEL_ROLE_USER, new Object[]{roleId});
        this.delete(new S_role(roleId));
    }

    public void execInsertRole(S_role s_role, String[] menuIds){
        this.insert(s_role);
        this.batchInsertRoleMenu(s_role.getRole_id(), menuIds);
    }

    public void execUpdateRole(S_role s_role, String[] menuIds){
        // 先删除角色关联的菜单功能，再写入新功能
        this.execute(SQL_DEL_ROLE_MENU, new Object[]{s_role.getRole_id()});
        this.batchInsertRoleMenu(s_role.getRole_id(), menuIds);
        this.save(s_role);
    }

    private void batchInsertRoleMenu(long roleId, String[] menuIdList){
        if(menuIdList == null || menuIdList.length == 0){
            return;
        }
        List<Object[]> parameters = new ArrayList<>(32);
        Object[] p = null;
        for(String menuId : menuIdList){
            p = new Object[2];
            p[0] = roleId;
            p[1] = menuId;
            parameters.add(p);
        }
        this.execBatchUpdate("insert into s_role_menu(role_id, menu_id, org_id) values(?,?,0)", parameters);
    }

    public void execUpdateStatus(long roleId, int status){
        this.execute(SQL_UPDATE_ROLE_STATUS, new Object[]{status, roleId});
    }

    private static final String SQL_ROLE_UNALLOCATED_USER = "select u.* from s_user_core u LEFT JOIN s_role_user ru ON ru.role_id=:roleId and ru.user_id=u.id where u.org_id=:orgId and ru.user_id is null and u.del_flag=0 and u.status=0";

    /**
     * 分页返回给定角色ID未分配用户的列表。即:该单位下关于该角色，仍没有分配的用户。(角色已经决定哪个单位了)
     * @param roleId
     * @param orgId
     * @param loginId
     * @return
     * @date 2022-12-21
     */
    public GenericPager<S_user_core> queryUnAllocatedUserList(long roleId, long orgId, String loginId){
        Map<String, Object> parameters = new HashMap<>();
        StringBuilder sql = new StringBuilder(SQL_ROLE_UNALLOCATED_USER);
        parameters.put("roleId", roleId);
        parameters.put("orgId", orgId);

        if(StringUtils.isNotEmpty(loginId)){
            sql.append(" and u.user_name like :userName");
            parameters.put("userName", "%" + loginId + "%");
        }
        PageSearch pageSearch = ListPageContext.getPageSearch();
        return this.selectSplit(sql.toString(), parameters, pageSearch.getPageIndex(), pageSearch.getPageSize(), new S_user_core());
    }

    private static final String SQL_ROLE_ALLOCATED_USER = "select u.* from s_role_user ru, s_user_core u where ru.role_id=:roleId and ru.user_id=u.id";

    /**
     * 分页返回已分配(给定角色ID)用户列表
     * @param roleId 角色ID
     * @param loginId 查询的用户登录ID
     * @return
     * @date 2022-12-21
     */
    public GenericPager<S_user_core> queryAllocatedUserList(long roleId, String loginId){
        Map<String, Object> parameters = new HashMap<>();
        StringBuilder sql = new StringBuilder(SQL_ROLE_ALLOCATED_USER);
        parameters.put("roleId", roleId);
        if(StringUtils.isNotEmpty(loginId)){
            sql.append(" and u.user_name like :userName");
            parameters.put("userName", "%" + loginId + "%");
        }
        PageSearch pageSearch = ListPageContext.getPageSearch();
        return this.selectSplit(sql.toString(), parameters, pageSearch.getPageIndex(), pageSearch.getPageSize(), new S_user_core());
    }

    private static final String SQL_ROLE_MENU_ID = "select menu_id from s_role_menu where role_id=?";

    /**
     * 返回给定角色拥有的菜单ID集合。
     * @param roleId
     * @return
     * @date 2022-12-19
     */
    public List<String> queryRoleMenuIdList(long roleId){
        List<Map<String, Object>> listMap = this.sqlQueryListMap(SQL_ROLE_MENU_ID, new Object[]{roleId});
        if(StringUtils.isEmptyList(listMap)){
            return null;
        }
        List<String> menuIdList = new ArrayList<>(32);
        for(Map<String, Object> m : listMap){
            menuIdList.add(m.get("menu_id").toString());
        }
        return menuIdList;
    }

    /**
     * 返回给定角色已经关联的用户数量，当前在删除角色时判断是否关联用户。
     * @param roleId
     * @return
     * @date 2022-12-19
     */
    public int queryRoleUserSize(long roleId){
        return this.queryForInt("select count(role_id) size from s_role_user where role_id=?", new Object[]{roleId});
    }

    /**
     * 检索给定单位中，是否存在包含给定名称的角色。
     * @param orgId 顶级单位ID
     * @param roleName 角色名字
     * @return
     * @date 2022-12-19
     */
    public S_role queryRoleByName(long orgId, String roleName){
        List<S_role> list = this.select("select * from s_role where org_id = ? and role_name = ?", new Object[]{orgId, roleName}, new S_role());
        if(StringUtils.isEmptyList(list)){
            return null;
        }
        return list.get(0);
    }

    /**
     * 分页查询角色列表
     * @param orgId 顶级单位ID
     * @param status 状态: 0 正常，1 禁用
     * @param roleName 查询的角色名称
     * @return
     * @date 2022-12-15
     */
    public GenericPager<S_role> queryPageRoleList(long orgId, int status, String roleName){
        Map<String, Object> parameters = new HashMap<>();
        StringBuilder sql = new StringBuilder(SQL_PAGE_ROLE_PREFIX);
        parameters.put("orgId", orgId);

        if(status >= 0){
            sql.append(" and status=:status");
            parameters.put("status", status);
        }
        if(StringUtils.isNotEmpty(roleName)){
            sql.append(" and role_name like :roleName");
            parameters.put("roleName", "%" + roleName + "%");
        }
        PageSearch pageSearch = ListPageContext.getPageSearch();
        return this.selectSplit(sql.toString(), parameters, pageSearch.getPageIndex(), pageSearch.getPageSize(), new S_role());
    }

    /**
     * 返回给定顶级机构下的角色列表。
     * @param orgId
     * @return
     */
    public List<S_role> queryRoleList(long orgId){
        return this.select(SQL_ORG_ROLES, new Object[]{orgId}, new S_role());
    }

    /**
     * 返回给定用户具有的角色
     * @param userId
     * @return
     */
    public List<S_role> queryUserRoleList(long userId){
        return this.select(SQL_USER_ROLES, new Object[]{userId}, new S_role());
    }

    /**
     * 根据角色ID，查询对应用户ID集合。
     * @param roleId
     * @return
     * @date 2023-07-31
     */
    public List<Long> queryRoleUserIdList(long roleId){
        List<Map<String, Object>> list = this.select(SQL_QUERY_ROLE_USER, new Object[]{roleId});
        if(StringUtils.isEmptyList(list)){
            return null;
        }
        List<Long> userIdList = new ArrayList<>(list.size());
        for(Map<String, Object> map : list){
            userIdList.add(Long.parseLong(map.get("user_id").toString()));
        }
        return userIdList;
    }

    private static final String SQL_QUERY_ROLE_USER = "select user_id from s_role_user where role_id=?";
}
