package com.iplatform.base.service;

import com.iplatform.base.PlatformUserCallback;
import com.iplatform.base.callback.UserProfileCallback;
import com.iplatform.base.pojo.UserInfoRequest;
import com.iplatform.base.util.role.SystemRole;
import com.iplatform.model.po.S_role;
import com.iplatform.model.po.S_user_core;
import com.walker.db.page.GenericPager;
import com.walker.db.page.ListPageContext;
import com.walker.db.page.PageSearch;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.jdbc.service.BaseServiceImpl;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Service
public class UserServiceImpl extends BaseServiceImpl {

    public static final String SQL_QUERY_USER_ROLES = "select role_id from s_role_user where user_id = ?";
    public static final String SQL_QUERY_USER_LOGIN_WHERE = "where user_name = ?";
    public static final String SQL_QUERY_USER_PHONE_WHERE = "where phonenumber = ?";
    public static final String SQL_QUERY_USER_EMAIL_WHERE = "where email = ?";

    // 2023-01-26 后台管理用户，不包括：App和设备等其他用户。
    public static final String SQL_PAGE_ORG_USER = "select * from s_user_core where user_type<99 and org_id=:orgId";

    public static final String SQL_INSERT_ROLE_USER = "insert into s_role_user(user_id, role_id, org_id) values(?,?,0)";
    public static final String SQL_UPDATE_STATUS = "update s_user_core set status=? where id=?";
    public static final String SQL_DELETE_USER_ROLE = "delete from s_role_user where user_id=?";
    public static final String SQL_RESET_PASSWORD = "update s_user_core set password=? where id=?";

    /**
     * 后台(当前用户)修改用户基本信息：姓名 + 密码
     * @param request
     * @param userId
     * @param callback 平台定义的更新回调，业务系统可以配置一个。
     * @date 2023-08-05
     */
    public void execUpdateUserInfo(UserInfoRequest request, long userId, UserProfileCallback callback){
        S_user_core userCore = new S_user_core(userId);
        if(StringUtils.isNotEmpty(request.getRealName())){
            userCore.setNick_name(request.getRealName());
        }
        if(StringUtils.isNotEmpty(request.getPassword())){
            userCore.setPassword(request.getPassword());
            userCore.setModify_pwd(1);
        }
        this.update(userCore);
        if(callback != null){
            callback.onUpdateUser(userCore, request);
        }
    }

    /**
     * 强制修改密码，并更新状态。
     * @param userId
     * @param encryptPassword
     * @date 2023-08-05
     */
    public void execForceChangePassword(long userId, String encryptPassword){
        S_user_core userCore = new S_user_core(userId);
        userCore.setPassword(encryptPassword);
        userCore.setModify_pwd(1);  // 设置为已修改密码
        this.update(userCore);
    }

    public void execResetPassword(long userId, String encryptPassword){
        this.execute(SQL_RESET_PASSWORD, new Object[]{encryptPassword, userId});
    }

    /**
     * 删除一个用户。
     * @param userId
     * @date 2022-12-13
     */
    public void execDeleteUser(long userId, PlatformUserCallback platformUserCallback){
        this.execute(SQL_DELETE_USER_ROLE, new Object[]{userId});
        this.delete(new S_user_core(userId));
        if(platformUserCallback != null){
            platformUserCallback.onDeleteUser(userId);
        }
    }

    /**
     * 更新用户状态。
     * @param userId
     * @param status
     * @date 2022-12-13
     */
    public void execUpdateUserStatus(long userId, int status){
        this.execute(SQL_UPDATE_STATUS, new Object[]{status, userId});
    }

    /**
     * 写入新用户记录，如果选择的有角色列表也一起保存。
     * @param user_core
     * @param roleIds
     * @date 2022-12-13
     */
    public void execSaveUser(S_user_core user_core, Long[] roleIds, PlatformUserCallback platformUserCallback){
        if(roleIds != null && roleIds.length > 0){
            for(long roleId : roleIds){
                this.execute(SQL_INSERT_ROLE_USER, new Object[]{user_core.getId(), roleId});
            }
        }
        this.insert(user_core);
        if(platformUserCallback != null){
            platformUserCallback.onCreateNewUser(user_core);
        }
    }

    /**
     * 编辑更新用户以及所属角色。
     * @param user_core
     * @param roleIds
     * @date 2022-12-13
     */
    public void execUpdateUser(S_user_core user_core, Long[] roleIds, PlatformUserCallback platformUserCallback){
        /*// 先删除角色关联信息
        this.execute(SQL_DELETE_USER_ROLE, new Object[]{user_core.getId()});
        if(roleIds != null && roleIds.length > 0){
            for(long roleId : roleIds){
                this.execute(SQL_INSERT_ROLE_USER, new Object[]{user_core.getId(), roleId});
            }
        }*/
        this.execUpdateAuthRole(user_core.getId(), roleIds);
        this.save(user_core);
        if(platformUserCallback != null){
            platformUserCallback.onUpdateUser(user_core);
        }
    }

    /**
     * 更新用户归属角色记录，如果给定的角色集合为空，则表示清除角色。
     * @param userId 用户ID
     * @param roleIds 给定角色ID集合，空表示要清除用户角色
     * @date 2022-12-15
     */
    public void execUpdateAuthRole(long userId, Long[] roleIds){
        // 先删除角色关联信息
        this.execute(SQL_DELETE_USER_ROLE, new Object[]{userId});
        if(roleIds != null && roleIds.length > 0){
            for(long roleId : roleIds){
                this.execute(SQL_INSERT_ROLE_USER, new Object[]{userId, roleId});
            }
        }
    }

    /**
     * 根据邮箱查询用户是否存在，如果存在邮箱存在的话。
     * @param email
     * @return
     * @date 2022-12-13
     */
    public S_user_core queryUserByEmail(String email){
        List<S_user_core> list = this.select(new S_user_core(), SQL_QUERY_USER_EMAIL_WHERE, new Object[]{email});
        if(StringUtils.isEmptyList(list)){
            return null;
        }
        return list.get(0);
    }

    /**
     * 根据手机号查询用户是否存在，如果存在手机号的话。
     * @param phoneNumber
     * @return
     * @date 2022-12-13
     */
    public S_user_core queryUserByPhone(String phoneNumber){
        List<S_user_core> list = this.select(new S_user_core(), SQL_QUERY_USER_PHONE_WHERE, new Object[]{phoneNumber});
        if(StringUtils.isEmptyList(list)){
            return null;
        }
        return list.get(0);
    }

    /**
     * 根据登录ID查询用户对象。
     * @param loginId
     * @return
     * @date 2022-12-13
     */
    public S_user_core queryUserByLoginId(String loginId){
        List<S_user_core> list = this.select(new S_user_core(), SQL_QUERY_USER_LOGIN_WHERE, new Object[]{loginId});
        if(StringUtils.isEmptyList(list)){
            return null;
        }
        return list.get(0);
    }

    /**
     * 分页查询用户列表。
     * @param orgId 顶级机构ID，必填
     * @param deptId 选择的部门ID，可选
     * @param loginId 登录用户ID
     * @param phoneNumber 手机号
     * @param status 状态
     * @return
     * @date 2022-12-12
     */
    public GenericPager<S_user_core> queryPageUserList(long orgId
            , long deptId, String loginId, String phoneNumber, int status){
        Map<String, Object> parameters = new HashMap<>();
        StringBuilder sql = new StringBuilder(SQL_PAGE_ORG_USER);
        parameters.put("orgId", orgId);

        if(deptId > 0){
            sql.append(" and dept_id = :deptId");
            parameters.put("deptId", deptId);
        }
        if(StringUtils.isNotEmpty(loginId)){
            sql.append(" and user_name = :userName");
            parameters.put("userName", loginId);
        }
        if(StringUtils.isNotEmpty(phoneNumber)){
            sql.append(" and phonenumber like :phoneNumber");
            parameters.put("phoneNumber", "%" + phoneNumber + "%");
        }
        if(status >= 0){
            sql.append(" and status=:status");
            parameters.put("status", status);
        }
        PageSearch pageSearch = ListPageContext.getPageSearch();
        return this.selectSplit(sql.toString(), parameters, pageSearch.getPageIndex(), pageSearch.getPageSize(), new S_user_core());
    }

    /**
     * 根据用户登录ID，查询用户信息以及包含的角色ID集合。
     * @param loginId
     * @param isPhone 是否手机号
     * @return 返回数组，userInfo[0] = S_user_core, userInfo[1] = roleIdList
     * @date 2022-11-06
     * @date 2023-06-28 增加对手机号登录支持，添加参数：isPhone
     */
    public Object[] queryLoginUser(String loginId, boolean isPhone){
//        this.queryForObject(SQL_QUERY_USER_LOGIN, new Object[]{loginId}, S_user_core.class);
        List<S_user_core> list = null;
        if(isPhone){
            list = this.select(new S_user_core(), SQL_QUERY_USER_PHONE_WHERE, new Object[]{loginId});
        } else {
            list = this.select(new S_user_core(), SQL_QUERY_USER_LOGIN_WHERE, new Object[]{loginId});
        }
        if(StringUtils.isEmptyList(list)){
            return null;
        }
        Object[] userInfo = new Object[2];
        S_user_core userCore = list.get(0);
        List<String> roleIdList = this.queryUserRoleIdList(userCore.getId());
        userInfo[0] = userCore;
        userInfo[1] = roleIdList;
        return userInfo;
    }

    /**
     * 返回登录用户对象
     * @param loginId 登录ID
     * @return
     * @date 2023-01-26 App登录时仅查询用户，不要角色信息。
     */
    public S_user_core queryLoginUserOnly(String loginId){
        List<S_user_core> list = this.select(new S_user_core(), SQL_QUERY_USER_LOGIN_WHERE, new Object[]{loginId});
        if(StringUtils.isEmptyList(list)){
            return null;
        }
        return list.get(0);
    }

    private static final String SQL_USER_DATASCOPE_MENU_LIST = "select DISTINCT(menu_id) from s_role_menu where role_id in (:roleIds) and menu_id like 'data_scope,%'";

    /**
     * 返回用户包含的菜单数据权限集合。
     * @param roleIds 用户具有的角色ID集合
     * @return 返回权限集合，即:menu_id list
     * @date 2022-12-21
     */
    public List<String> queryUserDataScopeMenuIdList(List<String> roleIds){
        Map<String, Object> parameters = new HashMap<>(2);
        parameters.put("roleIds", roleIds);
        List<Map<String, Object>> list = this.select(SQL_USER_DATASCOPE_MENU_LIST, parameters);
        if(StringUtils.isEmptyList(list)){
            return null;
        }
        List<String> menuIdList = new ArrayList<>(8);
        for(Map<String, Object> map : list){
            menuIdList.add(map.get("menu_id").toString());
        }
        return menuIdList;
    }

    /**
     * 查询给定用户所属的角色ID集合，角色ID不会重复。
     * @param userId
     * @return
     * @date 2022-11-06
     */
    public List<String> queryUserRoleIdList(long userId){
        List<Map<String, Object>> list = this.select(SQL_QUERY_USER_ROLES, new Object[]{userId});
        if(StringUtils.isEmptyList(list)){
            return null;
        }
        List<String> roleIdList = new ArrayList<>(4);
        for(Map<String, Object> m : list){
            roleIdList.add(m.get("role_id").toString());
        }
        return roleIdList;
    }

    /**
     * 返回用户授权角色列表，该列表包含独立单位下所有角色，在界面上展示哪些授权被选中。
     * @param orgId 顶级机构ID
     * @param userId 用户ID
     * @return
     * @date 2022-12-15
     */
    public List<SystemRole> queryAuthRoleList(long orgId, long userId){
        List<S_role> orgRoleList = this.select(SQL_ORG_ROLES, new Object[]{orgId}, new S_role());
        if(StringUtils.isEmptyList(orgRoleList)){
            return null;
        }
        List<SystemRole> systemRoleList = new ArrayList<>(8);
        List<String> userRoleIdList = this.queryUserRoleIdList(userId);
        if(!StringUtils.isEmptyList(userRoleIdList)){
            SystemRole systemRole = null;
            for(S_role role : orgRoleList){
                systemRole = new SystemRole(role);
                if(userRoleIdList.contains(String.valueOf(role.getRole_id()))){
                    systemRole.setFlag(true);
                }
                systemRoleList.add(systemRole);
            }
        } else {
            // 用户没有归属角色，需要把机构下所有角色展示出来
            for(S_role role : orgRoleList){
                systemRoleList.add(new SystemRole(role));
            }
        }
        return systemRoleList;
    }

    private static final String SQL_ORG_ROLES = "select * from s_role where org_id=? and status=0 and del_flag=0";

    /**
     * 根据用户ID返回所属角色对象集合。
     * @param userId
     * @return
     * @date 2022-12-13
     */
    @Deprecated
    public List<S_role> queryUserRoleList(long userId){
        return this.select(SQL_USER_ROLE_lIST, new Object[]{userId}, new S_role());
    }

    private static final String SQL_USER_ROLE_lIST = "select r.* from s_role_user ru, s_role r where ru.user_id=? and ru.role_id=r.role_id";
}
