package com.iplatform.base.support;

import com.iplatform.base.Constants;
import com.iplatform.base.service.ConfigArgumentServiceImpl;
import com.iplatform.base.util.ArgumentsManagerUtils;
import com.iplatform.core.SimpleVariable;
import com.iplatform.model.po.S_config;
import com.walker.cache.Cache;
import com.walker.infrastructure.arguments.ArgumentsManager;
import com.walker.infrastructure.arguments.Group;
import com.walker.infrastructure.arguments.Variable;
import com.walker.infrastructure.arguments.VariableType;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.security.SystemLogMan;
import com.walker.support.redis.cache.RedisCacheProvider;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 基于 Redis 实现的系统可变参数管理器，负责初始化并获取配置参数。
 * @author 时克英
 * @date 2022-11-29
 */
public class RedisArgumentsManager extends RedisCacheProvider<SimpleVariable> implements ArgumentsManager {

    private ConfigArgumentServiceImpl configArgumentService = null;

    // 2023-05-16，按照formId分组配置集合
    private Map<String, List<Variable>> formGroupReference = new HashMap<>();

    public void setConfigArgumentService(ConfigArgumentServiceImpl configArgumentService) {
        this.configArgumentService = configArgumentService;
        SystemLogMan.getInstance().checkMan();
    }

    public RedisArgumentsManager(){
        SystemLogMan.getInstance().checkMan();
        this.setUseRedis(true);
        this.setLoadPage(false);
    }

    @Override
    protected int loadDataToCache(Cache cache) {
        List<S_config> data = this.configArgumentService.selectAll(new S_config());
//        logger.info("+++++++++++++ 开始加载 argument");
        if(!StringUtils.isEmptyList(data)){
            if(this.isUseRedis()){
                // 如果redis中缓存数量和数据库中不一致（少），则清空redis缓存，重新加载数据库数据到缓存中。
                long totalCache = cache.getPersistentSize();
                if(totalCache != data.size()){
                    logger.info("redis缓存中【S_config】数量小于实际数据库中，需要清空缓存重新加载! cache = " + totalCache + ", db = " + data.size());
                    cache.clear();
                    for(S_config h : data){
                        cache.put(String.valueOf(h.getConfig_key()), ArgumentsManagerUtils.toVariable(h.getConfig_key(), h.getConfig_value(), h.getData_type()));
                    }
                }
            }

            // 2023-05-16，对于redis来说，只要系统启动，就需要重新加载分组信息，因为分组没有放入redis，而是在本机内存
            // 这在集群环境下，需要重新加载缓存对象，如：缓存管理中。
            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            Variable variable = null;
            List<Variable> groupList = null;
            for(S_config h : data){
                variable = ArgumentsManagerUtils.toVariable(h.getConfig_key(), h.getConfig_value(), h.getData_type());
                groupList = this.formGroupReference.get(String.valueOf(h.getForm_id()));
                if(groupList == null){
                    groupList = new ArrayList<>(8);
                    this.formGroupReference.put(String.valueOf(h.getForm_id()), groupList);
                }
                groupList.add(variable);
            }
            //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

            return data.size();
        }
        return 0;
    }

//    private SimpleVariable toVariable(String key , String value, String dataType){
//        SimpleVariable variable = new SimpleVariable();
//        variable.setId(key);
//        variable.setStringValue(value);
//        variable.setDataType(dataType);
//        return variable;
//    }

    @Override
    public SimpleVariable getVariable(String id) {
        return this.getCacheData(id);
    }

    @Override
    public void persist(String variableId, Object value) {
//        logger.warn("这些方法暂时不需要实现，因为后续配置参数管理后台不会使用:ArgumentsManager,直接读取数据库，业务也只需要使用获取方法!");
        SimpleVariable variable = ArgumentsManagerUtils.toVariable(variableId, value.toString(), VariableType.DEF_STRING);
        this.updateCacheData(variableId, variable);
    }

    @Override
    public void persist(String groupId, String variableId, Object value) {
        this.persist(variableId, value);
    }

    @Override
    public void persist(List<Object[]> changedList) {
        throw new UnsupportedOperationException("未实现代码");
    }

    @Override
    public void insert(List<Object[]> insertList) {
        throw new UnsupportedOperationException("未实现代码");
    }

    @Override
    public List<Group> getGroupList() {
        throw new UnsupportedOperationException("未实现代码");
    }

    @Override
    public List<Variable> getVariableList(String groupId) {
//        List<S_config> configList = this.configArgumentService.queryGroupConfigList(groupId);
//        return ArgumentsManagerUtils.acquireGroupVariableList(configList);
        // 使用新机制，根据formId返回分组参数集合。
        return this.formGroupReference.get(groupId);
    }

    @Override
    public void setSource(Object source) {
    }

    @Override
    public void destroy() throws Exception {

    }

    @Override
    public void afterPropertiesSet() throws Exception {
        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        // ~ 一定要调用父对象方法，2022-11-29
        // ~ 因为 RedisCacheProvider 和 ArgumentsManager 都有这个方法实现，否则初始load方法不会被加载!
        // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        super.afterPropertiesSet();
    }

    @Override
    public String getProviderName() {
        return Constants.CACHE_NAME_ARGUMENTS;
    }

    @Override
    public Class<?> getProviderType() {
        return SimpleVariable.class;
    }
}
