package com.iplatform.base.util;

import com.walker.infrastructure.utils.StringUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

public class DataImportUtils {

    /**
     * 计算要写入的数据集合，给定一个已存在的数据集合，判断当前 mapList 中哪些是不存在并需要新写入的数据。<p></p>
     * 目前使用简单的 Map 缓存已存在记录，并使用条件列值拼接字符串作为key，然后遍历 mapList筛查出来。
     * @param mapList 给定的原始数据集合
     * @param existList 已存在集合
     * @param whereColumnNames 条件字段名称集合
     * @return 返回结果，数组中包含两个集合: object[0] = insertList, object[1] = updateList
     * @date 2023-02-06
     */
    public static final Object[] calculateInsertAndUpdateList(List<Map<String, Object>> mapList
            , List<Map<String, Object>> existList, List<String> whereColumnNames){
        Object[] resultObject = new Object[2];

        // 只有写入集合
        if(StringUtils.isEmptyList(existList)){
            resultObject[0] = mapList;
            resultObject[1] = null;
            return resultObject;
        }
        Map<String, Integer> recordExistMap = new HashMap<>();
        StringBuilder key = null;
        for(Map<String, Object> existOne : existList){
            key = new StringBuilder();
            for(String whereColumn : whereColumnNames){
                key.append(existOne.get(whereColumn).toString());
            }
            recordExistMap.put(key.toString(), 1);
        }

        List<Map<String, Object>> insertList = new ArrayList<>(256);
        List<Map<String, Object>> updateList = new ArrayList<>(64);

        for(Map<String, Object> one : mapList){
            key = new StringBuilder();
            for(String whereColumn : whereColumnNames){
                key.append(one.get(whereColumn).toString());
            }
            // 不存在该更新记录，就属于直接写入的
            if(recordExistMap.get(key.toString()) == null){
                insertList.add(one);
            } else {
                updateList.add(one);
            }
        }
        resultObject[0] = insertList;
        resultObject[1] = updateList;
        return resultObject;
    }

    /**
     * 返回组装好的 where in (:param) 参数值集合，字符串类型的。
     * @param fieldName 查找条件的字段名称
     * @param mapList 原始数据Map集合。
     * @return
     * @date 2023-02-06
     */
    public static final List<String> acquireWhereInStringValues(String fieldName, List<Map<String, Object>> mapList){
        List<String> valueList = new ArrayList<>(256);
        Object value = null;
        for(Map<String, Object> map : mapList){
            value = map.get(fieldName);
            if(value == null){
                valueList.add(StringUtils.EMPTY_STRING);
            } else {
                valueList.add(value.toString());
            }
        }
        return valueList;
    }

    /**
     * 返回更新数据时，所有更新记录的SQL中的值几何。<p></p>
     * 如: [object[5], object[5], object[5]]
     * @param dataList
     * @param updateColumns
     * @param whereColumns
     * @return
     * @date 2023-02-05
     */
    public static final List<Object[]> acquireUpdateValues(List<Map<String, Object>> dataList
            , List<String> updateColumns, List<String> whereColumns){
        List<Object[]> list = new ArrayList<>();
        int allValueFieldSize = updateColumns.size() + whereColumns.size();

        Object[] parameter = null;
        for(Map<String, Object> map : dataList){
            parameter = new Object[allValueFieldSize];
            Object value = null;
            int currentPosition = 0;
            for(int i=0; i< updateColumns.size(); i++){
                // 遍历每个设置字段，根据字段名字，取出字段值
                value = map.get(updateColumns.get(i));
                parameter[currentPosition] = value;
                currentPosition ++;
            }

            // where 条件值
            for(int j=0; j< whereColumns.size(); j++){
                value = map.get(whereColumns.get(j));
                parameter[currentPosition] = value;
                currentPosition ++;
            }

            list.add(parameter);
        }
        return list;
    }

    /**
     * 生成更新SQL语句
     * @param tableName 表名
     * @param updateColumns 更新字段名集合
     * @param whereColumns 条件字段名集合，目前只能 and
     * @return 返回拼接的SQL语句，占位符使用:?
     * @date 2023-02-05
     */
    public static final String acquireUpdateSql(String tableName, List<String> updateColumns, List<String> whereColumns){
        StringBuilder sql = new StringBuilder("update ").append(tableName);
        sql.append(" set ");
        for(int i=0; i< updateColumns.size(); i++){
            if(i > 0){
                sql.append(StringUtils.DEFAULT_SPLIT_SEPARATOR);
            }
            sql.append(updateColumns.get(i)).append(StringUtils.CHAR_EQUALS).append("?");
        }
        if(!StringUtils.isEmptyList(whereColumns)){
            sql.append(" where ");
            for(int i=0; i< whereColumns.size(); i++){
                if(i > 0){
                    sql.append(" and ");
                }
                sql.append(whereColumns.get(i)).append(StringUtils.CHAR_EQUALS).append("?");
            }
        }
        return sql.toString();
    }

    /**
     * 把 Map 集合中数据转换成数组集合，方便 SQL 写入。
     * @param mapList
     * @param fieldNames
     * @return
     * @date 2023-02-06
     */
    public static final List<Object[]> translateToArray(List<Map<String, Object>> mapList, List<String> fieldNames){
        List<Object[]> list = new ArrayList<>(256);
        Object[] parameter = null;
        for(Map<String, Object> map : mapList){
            parameter = new Object[fieldNames.size()];
            for(int i=0; i< fieldNames.size(); i++){
                parameter[i] = map.get(fieldNames.get(i));
            }
            list.add(parameter);
        }
        return list;
    }

    /**
     * 把数组集合列表数据转换成Map集合。
     * @param dataList
     * @param fieldNames 字段名称集合
     * @return 注意:返回的Map中字段仍然按顺序，为: TreeMap
     * @date 2023-02-03
     */
    public static final List<Map<String, Object>> translateToMap(List<Object[]> dataList, List<String> fieldNames){
        if(StringUtils.isEmptyList(dataList)){
            return null;
        }
        List<Map<String, Object>> list = new ArrayList<>(dataList.size() + 2);
        Map<String, Object> map = null;
        for(Object[] row : dataList){
            map = new TreeMap<>();
            for(int i=0; i< row.length; i++){
                map.put(fieldNames.get(i), row[i]);
            }
            list.add(map);
        }
        return list;
    }

    /**
     * 根据表名、字段信息，生成一个 insert sql语句。
     * @param tableName
     * @param fieldNames
     * @return
     * @date 2023-02-01
     */
    public static final String acquireInsertSql(String tableName, List<String> fieldNames){
        StringBuilder sql = new StringBuilder("insert into ");
        sql.append(tableName).append("(");
        for(int i=0; i<fieldNames.size(); i++){
            if(i > 0){
                sql.append(StringUtils.DEFAULT_SPLIT_SEPARATOR);
            }
            sql.append(fieldNames.get(i));
        }
        sql.append(") values(");

        for(int i=0; i< fieldNames.size(); i++){
            if(i > 0){
                sql.append(StringUtils.DEFAULT_SPLIT_SEPARATOR);
            }
            sql.append("?");
        }
        sql.append(")");
        return sql.toString();
    }
}
