package com.iplatform.base.util;

import com.iplatform.base.VariableConstants;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.web.Constants;
import com.walker.web.TokenGenerator;
import jakarta.servlet.http.HttpServletRequest;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TokenUtils {

    private static final transient Logger logger = LoggerFactory.getLogger(TokenUtils.class);

    private static final long SECONDS_FOR_ONE_MINUTE = 60;

//    private static final long MINUTE_FOR_TOKEN_REFRESH_LEFT = 10;   // token剩余10分钟时提示前端刷新token

    /**
     * 缓存登录分钟数，应当比token时间少几分钟(默认5)，这样保证token失效前，redis清除登录信息，<br>
     * 系统能通知终端刷新token。
     * @date 2022-11-15
     */
    private static final long MINUTE_FOR_CACHE_LOGIN_LESS = 5;

    /**
     * 生成一个token，统一方法调用，业务都调用该方法。
     * @param userId 用户ID
     * @param loginId 用户登录名
     * @param uuid 生成的随机数
     * @param tokenGenerator
     * @param expiredMinutes 失效分钟数
     * @return
     * @date 2022-11-15
     * @date 2023-03-28 添加失效时间参数: expiredMinutes
     */
    public static final String generateToken(String userId
            , String loginId, String uuid, TokenGenerator tokenGenerator, long expiredMinutes){
        StringBuilder tokenData = new StringBuilder(userId);
        tokenData.append(StringUtils.DEFAULT_SPLIT_SEPARATOR);
        tokenData.append(loginId);
        return tokenGenerator.createToken(uuid, tokenData.toString()
//                , VariableConstants.DEFAULT_TOKEN_EXPIRED_MINUTES
                , expiredMinutes
                , VariableConstants.TOKEN_SECRET);
    }

//    /**
//     * 判断是否重新获取(刷新)token
//     * @param tokenExpiredMinutes 配置的token过期时间(分钟)
//     * @param tokenCreateMills token创建毫秒值
//     * @return
//     * @date 2022-11-14
//     */
//    public static final boolean isTokenRefresh(long tokenExpiredMinutes, long tokenCreateMills){
//        long currentTime = System.currentTimeMillis();
//        if(currentTime < tokenCreateMills){
//            logger.error("当前时间不能大于 'tokenCreateMills'");
//            return false;
//        }
//        long durationSeconds = (currentTime - tokenCreateMills) / 1000;
//        long tokenExpiredLeftMinute = tokenExpiredMinutes - durationSeconds/60;
//        if(tokenExpiredLeftMinute <= 0){
//            logger.warn("token已经过期，无需'refreshToken'");
//            return false;
//        }
//        if(tokenExpiredLeftMinute < MINUTE_FOR_TOKEN_REFRESH_LEFT){
//            return true;
//        }
//        return false;
//    }

    /**
     * 获得缓存用户的过期时间，该时间要比token生成过期时间略长(5分钟)。
     * @param tokenExpiredMinutes token过期分钟数
     * @return 返回缓存用户时间(秒)
     * @date 2022-11-14
     */
    public static final long acquireCacheUserExpiredSeconds(long tokenExpiredMinutes){
        if(tokenExpiredMinutes <= 0){
            throw new IllegalArgumentException("tokenExpiredMinutes 必须大于0");
        }
        if(tokenExpiredMinutes < MINUTE_FOR_CACHE_LOGIN_LESS){
            throw new IllegalArgumentException("设置的token失效时间(分钟)过短:" + tokenExpiredMinutes + ", 请重新设置。");
        }
        return (tokenExpiredMinutes - MINUTE_FOR_CACHE_LOGIN_LESS) * SECONDS_FOR_ONE_MINUTE;
    }

    /**
     * 返回token中业务数据，用户缓存的key，以及用户ID + 登录ID(2022-11-15追加)
     * @param tokenData
     * @return array[0] = user_id, array[1] = login_id, array[2] = user_key
     */
    public static final String[] getUserIdAndKey(String tokenData){
        if(StringUtils.isEmpty(tokenData)){
            return null;
        }
        return tokenData.split(StringUtils.DEFAULT_SPLIT_SEPARATOR);
    }

    public static final String getAuthorizationToken(HttpServletRequest request){
        // Constants.TOKEN_HEADER
        return getAuthorizationToken(request, Constants.TOKEN_HEADER);
    }

    /**
     * 从HTTP请求中获取授权token，默认token名字：Authorization
     * @param request
     * @param tokenName token名字
     * @return
     * @author 时克英
     * @date 2022-11-02
     */
    public static final String getAuthorizationToken(HttpServletRequest request, String tokenName){
        String token = request.getHeader(tokenName);
        if(StringUtils.isEmpty(token)){
            // 如果header中不存在，就在参数中查找
            token = request.getParameter(tokenName);
        }
        if(StringUtils.isEmpty(token)){
            return null;
        }
        if(token.startsWith(Constants.TOKEN_PREFIX)){
            token = token.replace(Constants.TOKEN_PREFIX, StringUtils.EMPTY_STRING);
        }
        return token;
    }

}
