package com.iplatform.file;

import com.iplatform.base.ArgumentsConstants;
import com.iplatform.base.Constants;
import com.iplatform.base.FileOperateSpi;
import com.iplatform.base.config.FileProperties;
import com.iplatform.core.BeanContextAware;
import com.iplatform.file.util.FileStoreUtils;
import com.walker.file.FileInfo;
import com.walker.file.FileStoreType;
import com.walker.infrastructure.arguments.ArgumentsManager;
import com.walker.infrastructure.arguments.ElementNotFoundException;
import com.walker.infrastructure.arguments.Variable;
import com.walker.infrastructure.utils.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.InputStream;
import java.util.List;

/**
 * 默认的文件操作服务提供者实现。
 * @author 时克英
 * @date 2023-02-15
 */
public class DefaultFileOperateSpi implements FileOperateSpi {

    protected final transient Logger logger = LoggerFactory.getLogger(getClass());

    private static final int FILE_CONTEXT_LENGTH = Constants.FILE_CONTEXT_PATH.length();

    /**
     * 清除给定文件地址的CDN前缀。如：https://qnyun.com/oss/12345678
     * <pre>
     *     去掉前缀后，只剩下"12345678"
     * </pre>
     * @param path 给定的文件资源地址
     * @return
     * @date 2023-05-17
     */
    @Override
    public String clearCdnPrefix(String path){
        if(StringUtils.isEmpty(path)){
            return path;
        }

        // 如果URL路径中包含本地识别部分（/file/）则直接替换，2023-06-10
        int fileIndex = path.indexOf(Constants.FILE_CONTEXT_PATH);
        if(fileIndex > 0){
            return path.substring(fileIndex + FILE_CONTEXT_LENGTH, path.length());
        }

        // 如果是其他远程存储方式，则根据存储类型分别获取配置前缀
        String prefix = this.getCdnUrl() + StringUtils.FOLDER_SEPARATOR;
        if(!prefix.endsWith(StringUtils.FOLDER_SEPARATOR)){
            prefix += StringUtils.FOLDER_SEPARATOR;
        }
        if(path.contains(prefix)){
            if (path.contains("callback/alipay")) {
                return path;
            }
            return path.replace(prefix, StringUtils.EMPTY_STRING);
        }
        return path;
    }

    /**
     * 获取上传文件的CDN地址，根据使用的不同第三方服务从配置中查找。
     * @return
     * @date 2023-05-17
     */
    @Override
    public String getCdnUrl(){
        Variable uploadTypeVar = this.acquireArgumentsManager().getVariable(ArgumentsConstants.CONFIG_UPLOAD_TYPE);
        if(uploadTypeVar == null){
            throw new ElementNotFoundException("参数未找到：" + ArgumentsConstants.CONFIG_UPLOAD_TYPE);
        }

        String uploadUrl = null;
        String uploadType = uploadTypeVar.getStringValue();
//        if(uploadType.equals("1") || uploadType.equals(FileStoreType.INDEX_FS)){
//            uploadUrl = ArgumentsConstants.CONFIG_LOCAL_UPLOAD_URL;
//        } else if(uploadType.equals("2") || uploadType.equals(FileStoreType.INDEX_OSS_QI_NIU)){
//            uploadUrl = ArgumentsConstants.CONFIG_QN_UPLOAD_URL;
//        } else if(uploadType.equals("3") || uploadType.equals(FileStoreType.INDEX_OSS_ALI)){
//            uploadUrl = ArgumentsConstants.CONFIG_AL_UPLOAD_URL;
//        } else if(uploadType.equals("4") || uploadType.equals(FileStoreType.INDEX_OSS_TX)){
//            uploadUrl = ArgumentsConstants.CONFIG_TX_UPLOAD_URL;
//        } else {
//            throw new UnsupportedOperationException("不支持的上传类型：" + uploadType);
//        }
        uploadUrl = FileStoreUtils.getFileUrlPrefixKey(uploadType);
        return this.acquireArgumentsManager().getVariable(uploadUrl).getStringValue();
    }

    /**
     * 上传文件你到本地磁盘。
     * @param inputStream
     * @param fileName
     * @param groupId
     * @param fileSize
     * @param businessType
     * @param owner
     * @return
     * @throws Exception
     * @date 2023-06-09
     */
    @Override
    public FileInfo uploadFileToLocal(InputStream inputStream
            , String fileName, String groupId, long fileSize, Integer businessType, String owner) throws Exception{
        return this.acquireFileEngineFactory().uploadFileToLocal(inputStream, fileName, groupId, fileSize, businessType, owner);
    }

    @Override
    public FileInfo[] uploadFileToLocal(InputStream[] inputStream, String[] fileName, String groupId, long[] fileSize, Integer businessType, String owner) throws Exception {
        return this.acquireFileEngineFactory().uploadFileToLocal(inputStream, fileName, groupId, fileSize, businessType, owner);
    }

    @Override
    public FileInfo uploadFileToFtp(InputStream inputStream
            , String fileName, String groupId, long fileSize, Integer businessType, String owner) throws Exception{
        return this.acquireFileEngineFactory().uploadFileToFtp(inputStream, fileName, groupId, fileSize, businessType, owner);
    }

    @Override
    public FileInfo[] uploadFileToFtp(InputStream[] inputStream, String[] fileName, String groupId, long[] fileSize, Integer businessType, String owner) throws Exception {
        return this.acquireFileEngineFactory().uploadFileToFtp(inputStream, fileName, groupId, fileSize, businessType, owner);
    }

    @Override
    public FileInfo uploadFileToOss(InputStream inputStream
            , String fileName, String groupId, long fileSize, Integer businessType, String owner, FileStoreType ossType) throws Exception{
        return this.acquireFileEngineFactory().uploadFileToOss(inputStream, fileName, groupId, fileSize, businessType, owner, ossType);
    }

    @Override
    public FileInfo[] uploadFileToOss(InputStream[] inputStream, String[] fileName, String groupId, long[] fileSize, Integer businessType, String owner, FileStoreType ossType) throws Exception {
        return this.acquireFileEngineFactory().uploadFileToOss(inputStream, fileName, groupId, fileSize, businessType, owner, ossType);
    }

    @Deprecated
    @Override
    public FileInfo uploadFileToSystem(InputStream inputStream, String fileName, String groupId, long fileSize) throws Exception {
        return this.acquireFileEngineFactory().uploadFileToSystem(inputStream, fileName, groupId, fileSize);
    }

    @Deprecated
    @Override
    public FileInfo uploadFileToFtp(InputStream inputStream, String fileName, String groupId, long fileSize) throws Exception {
        return this.acquireFileEngineFactory().uploadFileToFtp(inputStream, fileName, groupId, fileSize);
    }

    @Override
    public FileInfo getFileInfo(long id) {
        return this.acquireFileEngineFactory().getFileInfo(String.valueOf(id));
    }

    @Override
    public List<FileInfo> getFileInfoList(List<String> ids) {
        return this.acquireFileEngineFactory().getFileInfoList(ids);
    }

    @Override
    public String getFileRootConfig() {
        if(this.localFileRoot == null){
            this.localFileRoot = this.getFileProperties().getFileRoot();
        }
        return this.localFileRoot;
    }

    @Override
    public boolean isRemoteAsLocal() {
        if(this.remoteAsLocal == null){
            this.remoteAsLocal = this.getFileProperties().isRemoteAsLocal();
        }
        return this.remoteAsLocal;
    }

    private FileProperties getFileProperties(){
        return BeanContextAware.getBeanByType(FileProperties.class);
    }

    private FileEngineFactory acquireFileEngineFactory(){
        return BeanContextAware.getBeanByType(FileEngineFactory.class);
    }

    /**
     * 引入参数配置管理器。
     * @date 2023-05-17
     */
    private ArgumentsManager acquireArgumentsManager(){
        return BeanContextAware.getBeanByType(ArgumentsManager.class);
    }

    private Boolean remoteAsLocal = null;
    private String localFileRoot = null;

//    /**
//     * 引入参数配置管理器。
//     * @param argumentsManager
//     * @date 2023-05-17
//     */
//    public void setArgumentsManager(ArgumentsManager argumentsManager) {
//        this.argumentsManager = argumentsManager;
//    }
//
//    private ArgumentsManager argumentsManager;
}
