package com.iplatform.file.config;

import com.iplatform.base.config.FileProperties;
import com.iplatform.file.FileEngineFactory;
import com.iplatform.file.FileStoreCallback;
import com.iplatform.file.service.FileServiceImpl;
import com.iplatform.file.support.AliOssFileEngine;
import com.iplatform.file.support.AttachmentJdbcCallback;
import com.iplatform.file.support.DefaultFileSystemEngine;
import com.iplatform.file.support.DefaultFtpFileEngine;
import com.iplatform.file.support.QnOssFileEngine;
import com.iplatform.file.support.TxOssFileEngine;
import com.walker.file.FileStoreType;
import com.walker.file.ftp.FtpConfig;
import com.walker.infrastructure.arguments.ArgumentsManager;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

@Configuration
public class FileConfig {

    /**
     * 配置一个文件引擎工厂对象。
     * @param ftpFileEngine
     * @param fileSystemEngine
     * @return
     * @date 2023-02-14
     * @date 2023-07-03 添加属性：remoteAsLocal 应对本地测试没有FTP、OSS远程环境的情况。
     */
    @Bean
    public FileEngineFactory fileEngineFactory(DefaultFtpFileEngine ftpFileEngine, DefaultFileSystemEngine fileSystemEngine
            , QnOssFileEngine qnOssFileEngine, TxOssFileEngine txOssFileEngine, AliOssFileEngine aliOssFileEngine
            , ArgumentsManager argumentsManager, FileProperties fileProperties){
        FileEngineFactory fileEngineFactory = new FileEngineFactory();
        fileEngineFactory.register(FileStoreType.Ftp, ftpFileEngine);
        fileEngineFactory.register(FileStoreType.FileSystem, fileSystemEngine);
        fileEngineFactory.register(FileStoreType.OssQiNiu, qnOssFileEngine);
        fileEngineFactory.register(FileStoreType.OssTx, txOssFileEngine);
        fileEngineFactory.register(FileStoreType.OssAli, aliOssFileEngine);
        fileEngineFactory.setArgumentsManager(argumentsManager);
        fileEngineFactory.setRemoteAsLocal(fileProperties.isRemoteAsLocal());
        fileEngineFactory.startup();
        return fileEngineFactory;
    }

    /**
     * 阿里云 OSS 文件引擎定义。
     * @param fileProperties
     * @param fileStoreCallback
     * @return
     * @date 2023-06-09
     */
    @Bean
    public AliOssFileEngine aliOssFileEngine(FileProperties fileProperties, FileStoreCallback fileStoreCallback){
        AliOssFileEngine engine = new AliOssFileEngine();
        engine.setFileRoot(fileProperties.getFileRoot());
        engine.setFileStoreCallback(fileStoreCallback);
        return engine;
    }

    /**
     * 腾讯 OSS 文件引擎定义。
     * @param fileProperties
     * @param fileStoreCallback
     * @return
     * @date 2023-06-09
     */
    @Bean
    public TxOssFileEngine txOssFileEngine(FileProperties fileProperties, FileStoreCallback fileStoreCallback){
        TxOssFileEngine engine = new TxOssFileEngine();
        engine.setFileRoot(fileProperties.getFileRoot());
        engine.setFileStoreCallback(fileStoreCallback);
        return engine;
    }

    /**
     * 七牛云 OSS 文件引擎定义。
     * @param fileProperties
     * @param fileStoreCallback
     * @return
     * @date 2023-06-09
     */
    @Bean
    public QnOssFileEngine qnOssFileEngine(FileProperties fileProperties, FileStoreCallback fileStoreCallback){
        QnOssFileEngine engine = new QnOssFileEngine();
        engine.setFileRoot(fileProperties.getFileRoot());
        engine.setFileStoreCallback(fileStoreCallback);
        return engine;
    }

    /**
     * 本机文件系统 实现的文件管理引擎。
     * @param fileProperties
     * @param fileService
     * @return
     * @date 2023-02-14
     */
    @Bean
    public DefaultFileSystemEngine defaultFileSystemEngine(FileProperties fileProperties, FileServiceImpl fileService){
        DefaultFileSystemEngine fileSystemEngine = new DefaultFileSystemEngine();
        fileSystemEngine.setFileRoot(fileProperties.getFileRoot());
        fileSystemEngine.setFileStoreCallback(this.fileStoreCallback(fileService));
        return fileSystemEngine;
    }

    /**
     * Ftp 实现的文件管理引擎
     * @param ftpProperties
     * @param fileService
     * @return
     * @date 2023-02-14
     */
    @Bean
    public DefaultFtpFileEngine defaultFtpFileEngine(FtpProperties ftpProperties, FileServiceImpl fileService){
        FtpConfig ftpConfig = new FtpConfig();
        ftpConfig.setIp(ftpProperties.getIp());
        ftpConfig.setPort(ftpProperties.getPort());
        ftpConfig.setUserName(ftpProperties.getUserName());
        ftpConfig.setPassword(ftpProperties.getPassword());
        ftpConfig.setPrivateKey(ftpProperties.getPrivateKey());

        DefaultFtpFileEngine ftpFileEngine = new DefaultFtpFileEngine();
//        ftpFileEngine.setFileRoot(fileProperties.getFileRoot());
        ftpFileEngine.setFileRoot(ftpProperties.getFileRoot());
        ftpFileEngine.setFtpConfig(ftpConfig);
        ftpFileEngine.setFileStoreCallback(this.fileStoreCallback(fileService));
        return ftpFileEngine;
    }

    @Bean
    public FtpProperties ftpProperties(){
        return new FtpProperties();
    }

//    /**
//     * 放在 base spi 中配置。
//     * @return
//     */
//    @Bean
//    public FileProperties fileProperties(){
//        return new FileProperties();
//    }

    /**
     * 配置文件上传记录写入的回调实现。
     * @param fileService
     * @return
     * @date 2023-02-14
     */
//    @Bean
//    public JdbcCallback fileStoreCallback(FileServiceImpl fileService){
//        JdbcCallback jdbcCallback = new JdbcCallback();
//        jdbcCallback.setFileService(fileService);
//        return jdbcCallback;
//    }

    /**
     * 配置新的附件保存回调对象，废弃老对象：JdbcCallback
     * @param fileService
     * @return
     * @date 2023-06-05
     */
    @Bean
    public AttachmentJdbcCallback fileStoreCallback(FileServiceImpl fileService){
        AttachmentJdbcCallback jdbcCallback = new AttachmentJdbcCallback();
        jdbcCallback.setFileService(fileService);
        return jdbcCallback;
    }
}
