package com.iplatform.file.support;

import com.iplatform.base.util.UserUtils;
import com.iplatform.file.FileStoreCallback;
import com.iplatform.file.service.FileServiceImpl;
import com.iplatform.file.util.ImageUtils;
import com.iplatform.model.po.S_user_core;
import com.iplatform.model.po.SfAttachment;
import com.iplatform.model.po.SfAttachment_mapper;
import com.walker.file.DefaultFileInfo;
import com.walker.file.FileInfo;
import com.walker.file.FileStoreType;
import com.walker.infrastructure.utils.DateUtils;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.web.UserPrincipal;
import com.walker.web.WebRuntimeException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;

import java.util.ArrayList;
import java.util.List;

public class AttachmentJdbcCallback implements FileStoreCallback {

    protected final transient Logger logger = LoggerFactory.getLogger(getClass());

    @Override
    public void onCallback(FileInfo fileInfo, FileStoreType fileStoreType) {
        if(logger.isDebugEnabled()){
            logger.debug("保存文件:{}", fileInfo);
        }
        String fileExt = fileInfo.getFileExt();

        SfAttachment attachment = new SfAttachment();
        attachment.setAttId(Long.parseLong(fileInfo.getId()));
        attachment.setCreateTime(DateUtils.getDateTimeNumber());
        attachment.setUpdateTime(attachment.getCreateTime());

        // 2023-08-01
        try{
            UserPrincipal<S_user_core> userPrincipal = UserUtils.getCurrentUserPrincipal();
            if(userPrincipal != null){
                attachment.setUserName(userPrincipal.getUserName());
            } else {
                attachment.setUserName("前端用户");
            }
        } catch (WebRuntimeException ex){
            logger.debug("未找到当前用户，文件记录人为空");
            attachment.setUserName("未知用户");
        }

        attachment.setName(fileInfo.getFileName());
        attachment.setSattDir(fileInfo.getUrl());
        attachment.setAttType(fileExt);   // 这里目前是后缀名，后续可参考电商系统存放:mimeType
        attachment.setAttSize(fileInfo.getFileSize());
        attachment.setFileStoreType(fileStoreType.getIndex());
        if(StringUtils.isNotEmpty(fileInfo.getGroupId())){
            attachment.setGroupId(fileInfo.getGroupId());
        }
        if(ImageUtils.isImage(fileExt) || ImageUtils.isVideo(fileExt)){
            // 暂时存放为1，表示媒体格式
            attachment.setImageType(1);
        } else {
            // 非媒体格式
            attachment.setImageType(0);
        }
        // pid 和 owner 需要传过来
        attachment.setPid(fileInfo.getBusinessType());
        String owner = fileInfo.getOwner();
        if(StringUtils.isEmpty(owner)){
            attachment.setOwner(-1);
        } else {
            attachment.setOwner(Integer.parseInt(owner));
        }
        this.fileService.insert(attachment);
    }

    @Override
    public FileInfo onAcquireFileInfo(String id) {
        SfAttachment attachment = this.fileService.get(new SfAttachment(Long.parseLong(id)));
        if(attachment == null){
            return null;
        }
        return this.toFileInfo(attachment);
    }

    @Override
    public List<FileInfo> onAcquireFileInfoList(List<String> ids) {
        if(StringUtils.isEmptyList(ids)){
            return null;
        }
        MapSqlParameterSource sqlParameterSource = new MapSqlParameterSource();
        sqlParameterSource.addValue("ids", ids);
        List<SfAttachment> list = this.fileService.sqlListObjectWhereIn("select * from sf_attachment where att_id in(:ids)"
                , SfAttachment_mapper.ROW_MAPPER, sqlParameterSource);
        if(StringUtils.isEmptyList(list)){
            return null;
        }
        List<FileInfo> fileInfoList = new ArrayList<>(list.size());
        for(SfAttachment s_file : list){
            fileInfoList.add(this.toFileInfo(s_file));
        }
        return fileInfoList;
    }

    private FileInfo toFileInfo(SfAttachment attachment){
        if(attachment == null){
            return null;
        }
        DefaultFileInfo fileInfo = new DefaultFileInfo();
        fileInfo.setId(String.valueOf(attachment.getAttId()));
        fileInfo.setFileSize(attachment.getAttSize());
        fileInfo.setUrl(attachment.getSattDir());
        fileInfo.setFileExt(attachment.getAttType());
        fileInfo.setFileName(attachment.getName());
        fileInfo.setFileStoreType(FileStoreType.getType(attachment.getFileStoreType()));
        fileInfo.setGroupId(attachment.getGroupId());
        fileInfo.setBusinessType(attachment.getPid());
        fileInfo.setOwner(String.valueOf(attachment.getOwner()));
        return fileInfo;
    }

    public void setFileService(FileServiceImpl fileService) {
        this.fileService = fileService;
    }

    private FileServiceImpl fileService;

}
