package com.iplatform.generator.controller;

import com.fasterxml.jackson.databind.JsonNode;
import com.iplatform.base.SystemController;
import com.iplatform.generator.GenConstants;
import com.iplatform.generator.JdbcGeneratorEngine;
import com.iplatform.generator.config.JdbcGeneratorProperties;
import com.iplatform.generator.pojo.TableParam;
import com.iplatform.generator.service.CodeGenServiceImpl;
import com.iplatform.generator.util.CodeGenUtils;
import com.iplatform.model.po.S_gen_column;
import com.iplatform.model.po.S_gen_table;
import com.walker.db.TableInfo;
import com.walker.db.page.GenericPager;
import com.walker.db.page.ListPageContext;
import com.walker.db.page.PageSearch;
import com.walker.dbmeta.FieldInfo;
import com.walker.infrastructure.utils.JsonUtils;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.web.ResponseValue;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 代码生成界面管理接口。
 * @author 时克英
 * @date 2022-11-22
 */
@RestController
@RequestMapping("/tool/gen")
public class CodeGenController extends SystemController {

    public static final String PACKAGE_NAME = "com.iplatform.model.po";

    private CodeGenServiceImpl codeGenService;
    private JdbcGeneratorEngine jdbcGeneratorEngine;
    private JdbcGeneratorProperties jdbcGeneratorProperties;

    @Autowired
    public CodeGenController(CodeGenServiceImpl codeGenService
            , JdbcGeneratorEngine jdbcGeneratorEngine, JdbcGeneratorProperties jdbcGeneratorProperties){
        this.codeGenService = codeGenService;
        this.jdbcGeneratorEngine = jdbcGeneratorEngine;
        this.jdbcGeneratorProperties = jdbcGeneratorProperties;
    }

    @RequestMapping("/list")
    public ResponseValue listGenTable(S_gen_table genTable){
//        this.preparePageSearch();
        GenericPager<S_gen_table> pager = this.codeGenService.queryPageGenTable(genTable.getTable_name(), genTable.getTable_comment());
//        return this.acquireTablePage(pager.getDatas(), pager.getTotalRows());
        return ResponseValue.success(pager);
    }

    @RequestMapping("/db/list")
    public ResponseValue listDbTable(TableParam tableParam){
//        PageSearch pageSearch = this.preparePageSearch();
        PageSearch pageSearch = ListPageContext.getPageSearch();
        ListPageContext.setCurrentPageIndex(pageSearch.getPageIndex());
        ListPageContext.setCurrentPageSize(pageSearch.getPageSize());
        GenericPager<TableInfo> pager = this.jdbcGeneratorEngine.queryPageTableNamesByLike(tableParam.getTableName());
        logger.debug("tableInfo list = " + pager);
//        return this.acquireTablePage(pager.getDatas(), pager.getTotalRows());
        return ResponseValue.success(pager);
    }

    @RequestMapping("/import")
    public ResponseValue saveImportTable(String tables){
        if(StringUtils.isEmpty(tables)){
            return ResponseValue.error("请提供表名称");
        }
        String[] tableArray = StringUtils.toArray(tables);
        if(tableArray.length > 1){
            return ResponseValue.error("暂不支持多表同时导出，请选择一个");
        }
        logger.info("当前保存导入(生成)表 = " + tableArray[0]);
        TableInfo tableInfo = this.jdbcGeneratorEngine.queryOneTableInfo(tableArray[0]);
        if(tableInfo == null){
            return ResponseValue.error("选择的表不存在:" + tables);
        }

        String userName = this.getCurrentUser().getUser_name();
        List<FieldInfo> fieldList = this.jdbcGeneratorEngine.getFieldsObject(tableInfo.getName());
        logger.debug("fieldList = " + fieldList);

        String packageName = this.jdbcGeneratorProperties.getPackageName();
        if(StringUtils.isEmpty(packageName)){
            packageName = PACKAGE_NAME;
            logger.error("packageName 未找到配置项，使用默认值: " + PACKAGE_NAME);
        }
        S_gen_table s_gen_table = CodeGenUtils.createGenTable(tableInfo, userName, packageName);
        List<S_gen_column> columnList = new ArrayList<>();
        if(fieldList != null){
            for(FieldInfo fieldInfo : fieldList){
                S_gen_column s_gen_column = CodeGenUtils.createGenColumn(fieldInfo, userName, s_gen_table.getTable_id());
                columnList.add(s_gen_column);
            }
        }
        this.codeGenService.execInsertOneTableAndColumnList(s_gen_table, columnList);
        return ResponseValue.success();
    }

    /**
     * 重用 /tool/gen/db/** 的权限点，删除已生成的表记录。
     * @param tableIds
     * @return
     * @date 2022-11-26
     */
    @RequestMapping("/db/remove/{tableIds}")
    public ResponseValue removeGenTable(@PathVariable Long[] tableIds){
        if(tableIds == null || tableIds.length == 0){
            return ResponseValue.error("请提供表ID");
        }
        if(tableIds.length > 1){
            return ResponseValue.error("暂不支持多表删除，请选择一个");
        }
        this.codeGenService.execDeleteTableAndColumnList(tableIds[0]);
        return ResponseValue.success();
    }

    /**
     * 显示一个表结构信息，修改界面使用。
     * @param tableId
     * @return
     */
    @GetMapping(value = "/db/{tableId}")
    public ResponseValue showTableInfo(@PathVariable Long tableId){
        if(tableId == null | tableId.longValue() <= 0){
            return ResponseValue.error("缺少表参数");
        }
        S_gen_table s_gen_table = this.codeGenService.get(new S_gen_table(tableId));
        this.setTableFromOptions(s_gen_table);
        List<S_gen_column> columnList = this.codeGenService.queryGenColumnList(tableId);
        Map<String, Object> map = new HashMap<String, Object>(4);
        map.put("info", s_gen_table);
        map.put("rows", columnList);
        return ResponseValue.success(map);
    }

    @GetMapping("/db/gen_po/{tableName}")
    public void downloadGenPoZip(@PathVariable("tableName") String tableName){
        logger.debug(".......... tableName = " + tableName);
        if(StringUtils.isEmpty(tableName)){
            throw new IllegalArgumentException("未提供表明，无法下载生成PO");
        }
        byte[] data = this.jdbcGeneratorEngine.generateOnePoZipFile(tableName);
        try {
            this.downloadSimpleFile(data, tableName + ".zip");
        } catch (IOException e) {
            throw new RuntimeException("下载PO生成异常: " + tableName,e);
        }
    }

    private void setTableFromOptions(S_gen_table genTable){
        String options = genTable.getOptions();
        if(StringUtils.isNotEmpty(options)){
            try {
                JsonNode json = JsonUtils.jsonStringToObject(options, JsonNode.class);
                String treeCode = json.get(GenConstants.TREE_CODE).asText();
                String treeParentCode = json.get(GenConstants.TREE_PARENT_CODE).asText();
                String treeName = json.get(GenConstants.TREE_NAME).asText();
                String parentMenuId = json.get(GenConstants.PARENT_MENU_ID).asText();
                String parentMenuName = json.get(GenConstants.PARENT_MENU_NAME).asText();

                genTable.setTreeCode(treeCode);
                genTable.setTreeParentCode(treeParentCode);
                genTable.setTreeName(treeName);
                genTable.setParentMenuId(parentMenuId);
                genTable.setParentMenuName(parentMenuName);
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        }
    }
}
