package com.iplatform.openocr;

import com.walker.infrastructure.utils.StringUtils;
import com.walker.openocr.OcrType;
import com.walker.openocr.TextBlock;
import com.walker.openocr.TextResolver;
import com.walker.openocr.idcard.IdCardTextResolver;
import com.walker.openocr.table.TableConfig;
import com.walker.openocr.table.TableTextResolver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public abstract class AbstractOcrEngine implements OcrEngine{

    protected final transient Logger logger = LoggerFactory.getLogger(this.getClass());

    private String remoteUrl = null;

    private List<TableConfig> tableConfigList = new ArrayList<>(8);

    private Map<String, TextResolver<?,?>> textResolverMap = new HashMap<>();

    @Override
    public void setRemoteUrl(String url) {
        if(StringUtils.isEmpty(url)){
            throw new IllegalArgumentException("OCR远程服务不能为空");
        }
        this.remoteUrl = url;
    }

    @Override
    public TextResolver<?, ?> getTextResolver(OcrType ocrType) {
        TextResolver<?,?> textResolver = this.textResolverMap.get(ocrType.getIndex());
        if(textResolver == null){
            throw new IllegalArgumentException("不存在指定的文本解析器:" + ocrType);
        }
        return textResolver;
    }

    @Override
    public List<TextBlock> recognize(String id, String imgBase64) {
        return null;
    }

    @Override
    public void startup() {
        List<TableConfig> tableConfigs = this.loadTableConfigs();
        if(StringUtils.isEmptyList(tableConfigs)){
            logger.warn("未加载到任何 TableConfig");
        } else {
            this.tableConfigList.addAll(tableConfigs);
            logger.info("加载 TableConfig 数量:" + this.tableConfigList.size());
        }

        if(this.tableConfigList.size() > 0){
            TableTextResolver tableTextResolver = new TableTextResolver();
            this.textResolverMap.put(OcrType.TEXT_TABLE, tableTextResolver);
        }

        // 创建身份证解析器
        IdCardTextResolver idCardTextResolver = new IdCardTextResolver();
        this.textResolverMap.put(OcrType.TEXT_ID_CARD, idCardTextResolver);
    }

    protected String getRemoteUrl() {
        return remoteUrl;
    }

    @Override
    public List<TableConfig> getTableConfigList() {
        return tableConfigList;
    }

    protected abstract List<TableConfig> loadTableConfigs();
}
