package com.iplatform.scheduler;

import com.iplatform.model.po.S_scheduler;
import com.iplatform.scheduler.util.SchedulerUtils;

import com.walker.infrastructure.utils.StringUtils;
import com.walker.scheduler.GatherSchedulerManager;
import com.walker.scheduler.ScheduleEngine;
import com.walker.scheduler.impl.TimedScheduler;
import com.walker.security.SystemLogMan;

import java.util.List;

/**
 * 平台调度任务管理器抽象，可通过该对象管理任务列表。
 * 包括：启动、暂停、停止、列表等操作。
 * @author 时克英
 * @date 2022-09-08
 */
public abstract class PlatformSchedulerManager extends GatherSchedulerManager {

    private ScheduleEngine scheduleEngine;

    public void startup(){
        this.initialize();
    }

    @Override
    protected void initialize() {
        this.scheduleEngine = this.acquireScheduleEngine();
        if(this.scheduleEngine == null){
            throw new IllegalArgumentException("请提供一个可用的调度引擎对象: ScheduleEngine");
        }
        SystemLogMan.getInstance().checkMan();
        List<S_scheduler> schedulerList = this.acquireRunningList();
        int count = 0;

        if(!StringUtils.isEmptyList(schedulerList)){
            boolean result = false;
            for(S_scheduler e : schedulerList){
                if(e.getStatus() == Constants.STATUS_START){
                    result = this.runScheduler(e);
                    if(!result){
                        logger.info("一个调度器初始化运行失败:" + e.getId());
                    } else {
                        count++;
                    }

                } else if(e.getStatus() == Constants.STATUS_PAUSE){
                    this.putPauseScheduler(e);
                }
            }
        }
        logger.info("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~");
        logger.info("初始化了调度管理器，加载已有调度任务:" + count + "个");
        logger.info("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~");
    }

    protected boolean runScheduler(S_scheduler entity){
        try {
            TimedScheduler scheduler = SchedulerUtils.createScheduler(entity, this.scheduleEngine);
            this.startScheduler(scheduler);
            return true;
        } catch (Exception e) {
            logger.error("系统启动后，自动运行调度器错误：" + entity.getName() + "，id=" + entity.getId(), e);
            return false;
        }
    }

    protected TimedScheduler putPauseScheduler(S_scheduler e){
        try {
            TimedScheduler scheduler = SchedulerUtils.createScheduler(e, this.scheduleEngine);
            scheduler.pause();
            this.getCachedData().put(e.getId().intValue(), scheduler);
            return scheduler;
        } catch (Exception e1) {
            logger.error("系统启动后，创建调度器错误：" + e.getName() + "，id=" + e.getId(), e1);
            return null;
        }
    }

    public ScheduleEngine getScheduleEngine(){
        return this.scheduleEngine;
    }

    /**
     * 获取正在运行的调度任务（包括：暂停的）
     * @return
     */
    protected abstract List<S_scheduler> acquireRunningList();

    /**
     * 获得一个调度引擎对象，用于操作任务状态。
     * @return
     */
    protected abstract ScheduleEngine acquireScheduleEngine();

}
