package com.ishop.merchant.cache;

import com.iplatform.base.PlatformRuntimeException;
import com.ishop.merchant.CityCacheProvider;
import com.ishop.merchant.Constants;
import com.ishop.merchant.service.CityServiceImpl;
import com.ishop.merchant.util.CityUtils;
import com.ishop.model.po.EbCityRegion;
import com.ishop.model.vo.CityTreeVo;
import com.walker.cache.Cache;
import com.walker.infrastructure.utils.JsonUtils;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.support.redis.cache.RedisCacheProvider;

import java.util.ArrayList;
import java.util.List;

public class RedisCityCache extends RedisCacheProvider<CityTreeVo> implements CityCacheProvider {

    private static final String KEY_CITY_TREE = "city:tree";

    public RedisCityCache(){
        this.setUseRedis(true);
        this.setLoadPage(false);
    }

    @Override
    protected int loadDataToCache(Cache cache) {
        List<EbCityRegion> hosts = this.cityService.queryAllCityForCacheTree();
        if(!StringUtils.isEmptyList(hosts)){
            // ------------------------- 切换成普通缓存步骤：3
            if(this.isUseRedis()){
                // 如果redis中缓存数量和数据库中不一致（少），则清空redis缓存，重新加载数据库数据到缓存中。
                long totalCache = cache.getPersistentSize();
                // 2023-06-26 这里必须判定数据库中大于缓存中，因为缓存中多存了一个树对象（cache = 44996, db = 44995）
                if(totalCache < hosts.size()){
                    logger.info("redis缓存中用户数量小于实际用户，需要清空缓存重新加载! cache = " + totalCache + ", db = " + hosts.size());
                    cache.clear();

                    List<CityTreeVo> cityTreeVoList = new ArrayList<>(256);
                    CityUtils.buildTree(cityTreeVoList, hosts, cache);
                    logger.info("城市区域树根节点（数量）：" + cityTreeVoList.size());

                    try {
                        String cityTreeJson = JsonUtils.objectToJsonString(cityTreeVoList);
                        cache.put(KEY_CITY_TREE, cityTreeJson);
                        logger.info("添加城市树缓存成功：" + KEY_CITY_TREE);
                    } catch (Exception e) {
                        throw new PlatformRuntimeException("Redis缓存'cityTreeJson'发生错误:" + e.getMessage(), e);
                    }
                }
            }//------------------------------------------
            return hosts.size();
        }
        return 0;
    }

    @Override
    public List<CityTreeVo> getTree() {
        Object cityTreeJson = this.getCache().get(KEY_CITY_TREE);
        if(cityTreeJson == null || StringUtils.isEmpty(cityTreeJson.toString())){
            logger.error("缓存不存在：" + KEY_CITY_TREE);
            return null;
        }
        try {
            List<CityTreeVo> cityTreeVoList = JsonUtils.jsonStringToList(cityTreeJson.toString(), CityTreeVo.class);
            return cityTreeVoList;
        } catch (Exception e) {
            throw new PlatformRuntimeException("json字符串转List<CityTreeVo>错误:" + e.getMessage(), e);
        }
    }

    @Override
    public List<CityTreeVo> getList() {
        throw new UnsupportedOperationException("暂未实现代码");
    }

    @Override
    public CityTreeVo get(int id) {
        return this.getCacheData(String.valueOf(id));
    }

    @Override
    public void save(CityTreeVo category) {
        this.putCacheData(String.valueOf(category.getRegionId()), category);
    }

    @Override
    public void update(CityTreeVo category) {
        this.updateCacheData(String.valueOf(category.getRegionId()), category);
    }

    @Override
    public void remove(int id) {
        this.removeCacheData(String.valueOf(id));
    }

    @Override
    public String getProviderName() {
        return Constants.CACHE_NAME_CITY;
    }

    @Override
    public Class<?> getProviderType() {
        return CityTreeVo.class;
    }

    public void setCityService(CityServiceImpl cityService) {
        this.cityService = cityService;
    }

    private CityServiceImpl cityService;
}
