package com.ishop.merchant.controller;

import com.iplatform.base.Constants;
import com.iplatform.core.SimpleVariable;
import com.ishop.merchant.BaseController;
import com.ishop.merchant.NotifyConstants;
import com.ishop.merchant.OrderConstants;
import com.ishop.merchant.pojo.OrderParam;
import com.ishop.merchant.util.VoUtils;
import com.ishop.model.po.EbExpress;
import com.ishop.model.po.EbMerchantOrder;
import com.ishop.model.po.EbOrder;
import com.ishop.model.po.EbOrderDetail;
import com.ishop.model.po.EbOrderInvoice;
import com.ishop.model.po.EbOrderInvoiceDetail;
import com.ishop.model.request.MarkOrderRequest;
import com.ishop.model.request.OrderSendRequest;
import com.ishop.model.vo.OrderCountVo;
import com.ishop.model.vo.OrderInvoiceVo;
import com.ishop.model.vo.PlatformOrderDetailVo;
import com.ishop.model.vo.PlatformOrderVo;
import com.walker.db.page.GenericPager;
import com.walker.db.page.ListPageContext;
import com.walker.infrastructure.arguments.Variable;
import com.walker.infrastructure.time.TimeRange;
import com.walker.infrastructure.time.TimeRangeUtils;
import com.walker.infrastructure.utils.DateUtils;
import com.walker.infrastructure.utils.NumberGenerator;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.web.ResponseValue;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import java.util.ArrayList;
import java.util.List;

@RestController
@RequestMapping("/merchant/order")
public class MerchantOrderController extends BaseController {

    @RequestMapping(value = "/send", method = RequestMethod.POST)
    public ResponseValue send(@RequestBody OrderSendRequest request){
        if(request == null){
            return ResponseValue.error(Constants.ERROR_ARGUMENT);
        }
        if(StringUtils.isEmpty(request.getOrderNo())
                || StringUtils.isEmpty(request.getExpressNumber()) || StringUtils.isEmpty(request.getExpressCode())){
            return ResponseValue.error("快递信息不完整");
        }
        if (request.getDeliveryType().equals(OrderConstants.ORDER_DELIVERY_TYPE_EXPRESS)) {
            if (request.getExpressRecordType() == null) {
                return ResponseValue.error("请选择发货记录类型");
            }
            String error = validateExpressSend(request);
            if(error != null){
                return ResponseValue.error(error);
            }
        }

        int merId = this.getCurrentUser().getMer_id().intValue();
        EbOrder order = this.getOrderService().queryOrder(request.getOrderNo());
        if (order.getIsUserDel().intValue() == 1 || order.getIsMerchantDel().intValue() == 1) {
            return ResponseValue.error("订单已删除");
        }
//        if (order.getLevel().intValue() != OrderConstants.ORDER_LEVEL_MERCHANT) {
//            throw new CrmebException("订单等级异常，不是商户订单");
//        }
        if (order.getStatus().intValue() != OrderConstants.ORDER_STATUS_WAIT_SHIPPING
                && order.getStatus().intValue() != OrderConstants.ORDER_STATUS_PART_SHIPPING) {
            return ResponseValue.error("订单不处于待发货状态");
        }

        EbMerchantOrder merchantOrder = this.getOrderService().queryMerchantOrderOne(request.getOrderNo());
        if (merchantOrder.getShippingType().intValue() != OrderConstants.ORDER_SHIPPING_TYPE_EXPRESS) {
            return ResponseValue.error("订单非发货类型订单");
        }

        // 正常发货，非拆单
        if (!request.getIsSplit()) {
            if (request.getDeliveryType().equals(OrderConstants.ORDER_DELIVERY_TYPE_FICTITIOUS)) {
//                return virtual(order.getOrderNo());
                throw new UnsupportedOperationException("未实现虚拟发货");
            }
            if (request.getDeliveryType().equals(OrderConstants.ORDER_DELIVERY_TYPE_EXPRESS)) {
                return sendExpress(request, order, merchantOrder);
            }
        }

        // 拆单发货，暂不实现，2023-08-20
        return ResponseValue.error("还未实现拆单发货");
    }

    private ResponseValue sendExpress(OrderSendRequest request, EbOrder order, EbMerchantOrder merchantOrder){
        EbExpress express = this.getExpressCache().get(request.getExpressCode());
        if(express == null){
            return ResponseValue.error("快递公司不存在");
        }
        List<EbOrderDetail> orderDetailList = this.getOrderService().queryOrderDetailList(request.getOrderNo());
        List<EbOrderInvoiceDetail> orderInvoiceDetailList = new ArrayList<>();
        List<EbOrderDetail> orderDetailUpdateList = new ArrayList<>();
        for(EbOrderDetail od : orderDetailList){
            if (od.getPayNum() > od.getDeliveryNum()) {
                EbOrderInvoiceDetail invoiceDetail = VoUtils.acquireOrderInvoiceDetail(od);
                invoiceDetail.setNum(od.getPayNum() - od.getDeliveryNum());
                orderInvoiceDetailList.add(invoiceDetail);
                od.setDeliveryNum(od.getPayNum());
                orderDetailUpdateList.add(od);
            }
        }
        if(StringUtils.isEmptyList(orderInvoiceDetailList)){
            return ResponseValue.error("订单没有需要发货的商品");
        }

        String trackingNumber = request.getExpressNumber();
        // 电子面单，暂未集成，2023-08-21
//        if (request.getExpressRecordType().equals(2)) { // 电子面单
//            List<String> productNameList = orderInvoiceDetailList.stream().map(OrderInvoiceDetail::getProductName).collect(Collectors.toList());
//            String cargo = String.join(",", productNameList);
//            trackingNumber = expressDump(request, merchantOrder, express, cargo);
//        }

        // 生成发货单
        EbOrderInvoice orderInvoice = new EbOrderInvoice();
        orderInvoice.setId(NumberGenerator.getLongSequenceNumber());
        orderInvoice.setMerId(order.getMerId());
        orderInvoice.setOrderNo(order.getOrderNo());
        orderInvoice.setUid(order.getUid());
        orderInvoice.setExpressCode(express.getCode());
        orderInvoice.setExpressName(express.getName());
        orderInvoice.setTrackingNumber(trackingNumber);
        orderInvoice.setTotalNum(orderInvoiceDetailList.stream().mapToInt(EbOrderInvoiceDetail::getNum).sum());
        orderInvoice.setCreateTime(DateUtils.getDateTimeNumber());
        orderInvoice.setUpdateTime(orderInvoice.getCreateTime());

        for(EbOrderInvoiceDetail e : orderInvoiceDetailList){
            e.setInvoiceId(orderInvoice.getId());
        }
        // 这里不用设置，创建时已经默认快递：express方式
//        merchantOrder.setDeliveryType(OrderConstants.ORDER_DELIVERY_TYPE_EXPRESS);
        EbOrder updateOrder = new EbOrder(order.getId());
        updateOrder.setStatus(OrderConstants.ORDER_STATUS_WAIT_RECEIPT);

        // 通知的消息
        String message = OrderConstants.ORDER_LOG_MESSAGE_EXPRESS.replace("{deliveryName}", express.getName()).replace("{deliveryCode}", trackingNumber);
        this.getOrderService().execSendExpress(orderInvoice, orderInvoiceDetailList, orderDetailUpdateList, updateOrder);

        // 异步发送通知
        List<Variable> variableList = new ArrayList(8);
//        variableList.add(new SimpleVariable("first", "订单发货提醒"));
//        variableList.add(new SimpleVariable("keyword1", order.getOrderNo()));
//        variableList.add(new SimpleVariable("keyword2", orderInvoice.getExpressName()));
//        variableList.add(new SimpleVariable("keyword3", orderInvoice.getExpressCode()));
//        variableList.add(new SimpleVariable("remark", "欢迎再次光临"));
        variableList.add(new SimpleVariable("character_string2", order.getOrderNo()));
        variableList.add(new SimpleVariable("thing13", orderInvoice.getExpressName()));
        variableList.add(new SimpleVariable("character_string14", trackingNumber));
        variableList.add(new SimpleVariable("thing22", this.getMerchantCache().get(merchantOrder.getMerId()).getName()));
        this.pushNotificationWithTemplate(NotifyConstants.DELIVER_GOODS_MARK, variableList, order.getUid());
        return ResponseValue.success();
    }

    /**
     * 校验快递发货参数
     * @param request 发货参数
     */
    private String validateExpressSend(OrderSendRequest request) {
        if (StringUtils.isEmpty(request.getExpressCode())) return "请选择快递公司";
        if (request.getExpressRecordType().equals("1")) {
            if (StringUtils.isEmpty(request.getExpressNumber())){
                return "请填写快递单号";
            }
        } else  {
            // 电子面单，需要填写
            if (StringUtils.isEmpty(request.getExpressTempId())) return "请选择电子面单";
            if (StringUtils.isEmpty(request.getToName())) return "请填写寄件人姓名";
            if (StringUtils.isEmpty(request.getToTel())) return "请填写寄件人电话";
            if (StringUtils.isEmpty(request.getToAddr())) return "请填写寄件人地址";
        }
        return null;
    }

    /**
     * 订单细节详情列表（发货使用）
     * @return
     * @date 2023-08-21
     */
    @RequestMapping(value = "/detail/list", method = RequestMethod.GET)
    public ResponseValue getDetailList(String orderNo){
        if(StringUtils.isEmpty(orderNo)){
            return ResponseValue.error(Constants.ERROR_ARGUMENT);
        }
        return ResponseValue.success(this.getOrderService().queryShipmentList(orderNo));
    }

    /**
     * 商户备注订单
     * @param request
     * @return
     */
    @RequestMapping(value = "/mark", method = RequestMethod.POST)
    public ResponseValue mark(@RequestBody MarkOrderRequest request){
        if(request == null
                || StringUtils.isEmpty(request.getOrderNo()) || StringUtils.isEmpty(request.getRemark())){
            return ResponseValue.error("请输入订单备注");
        }
        EbMerchantOrder merchantOrder = this.getOrderService().queryMerchantOrderOne(request.getOrderNo());
        if(merchantOrder == null){
            return ResponseValue.error("商户订单不存在");
        }
        EbMerchantOrder updateOrder = new EbMerchantOrder(merchantOrder.getId());
        updateOrder.setMerchantRemark(request.getRemark());
        this.getOrderService().update(updateOrder);
        return ResponseValue.success();
    }

    /**
     * 获取订单发货单列表
     * @return
     * @date 2023-08-18
     */
    @RequestMapping(value = "/invoice/list", method = RequestMethod.GET)
    public ResponseValue getInvoiceList(String orderNo){
        if(StringUtils.isEmpty(orderNo)){
            return ResponseValue.error("订单编号不存在");
        }
        List<OrderInvoiceVo> list = this.acquireOrderInvoiceList(orderNo);
        return ResponseValue.success(list);
    }

    @RequestMapping(value = "/info", method = RequestMethod.GET)
    public ResponseValue info(String orderNo){
        if(StringUtils.isEmpty(orderNo)){
            return ResponseValue.error("订单编号不存在");
        }
        EbOrder order = this.getOrderService().queryOrder(orderNo);
        if(order == null){
            return ResponseValue.error("订单不存在");
        }

        int merId = this.getCurrentUser().getMer_id().intValue();
        if (order.getIsMerchantDel().intValue() == 1 || order.getMerId().intValue() != merId) {
            return ResponseValue.error("未找到对应订单信息");
        }
        PlatformOrderDetailVo vo = this.acquireCommonOrderDetail(order, orderNo);
        return ResponseValue.success(vo);
    }

    @RequestMapping(value = "/delete", method = RequestMethod.POST)
    public ResponseValue delete(String orderNo){
        if(StringUtils.isEmpty(orderNo)){
            return ResponseValue.error(Constants.ERROR_ARGUMENT);
        }
        int merId = this.getCurrentUser().getMer_id().intValue();
        EbOrder order = this.getOrderService().queryPlatformOrder(orderNo, merId);
        if(order == null){
            return ResponseValue.error("订单不存在");
        }
        if(order.getIsUserDel().intValue() != 1){
            return ResponseValue.error("用户还未删除订单");
        }
        EbOrder updateOrder = new EbOrder(order.getId());
        updateOrder.setIsMerchantDel(1);
        this.getOrderService().update(updateOrder);
        return ResponseValue.success();
    }

    /**
     * 平台端获取订单各状态数量
     * @return
     */
    @RequestMapping(value = "/status/num", method = RequestMethod.GET)
    public ResponseValue getOrderStatusNum(String dateLimit){
        OrderCountVo vo = new OrderCountVo();
        return ResponseValue.success(vo);
    }

    @RequestMapping(value = "/list", method = RequestMethod.GET)
    public ResponseValue list(OrderParam param){
        int merId = this.getCurrentUser().getMer_id().intValue();
        GenericPager<EbOrder> pager = null;
        if(param == null){
            pager = this.getOrderService().queryPagePlatformOrderList(merId, null, null, null, null);
        } else {
            TimeRange timeRange = StringUtils.isEmpty(param.getDateLimit())? null : TimeRangeUtils.getDateLimit(param.getDateLimit());
            pager = this.getOrderService().queryPagePlatformOrderList(merId, param.getOrderNo(), param.getType(), timeRange, param.getStatus());
        }

        List<EbOrder> data = pager.getDatas();
        if(StringUtils.isEmptyList(data)){
            return ResponseValue.success(pager);
        }
        List<PlatformOrderVo> list = this.acquireCommonOrderList(data);
        return ResponseValue.success(ListPageContext.createGenericPager(list, pager.getPageIndex(), (int)pager.getTotalRows()));
    }
}
