package com.ishop.merchant.controller;

import com.iplatform.base.Constants;
import com.iplatform.base.PlatformRuntimeException;
import com.iplatform.model.po.S_user_core;
import com.ishop.merchant.BaseController;
import com.ishop.merchant.ProductConstants;
import com.ishop.merchant.VariableConstants;
import com.ishop.merchant.pojo.BrandCategoryParam;
import com.ishop.merchant.pojo.CopyProductRequest;
import com.ishop.merchant.pojo.ProductParam;
import com.ishop.merchant.util.ImageUtils;
import com.ishop.merchant.util.ProductAttrUtils;
import com.ishop.merchant.util.VoUtils;
import com.ishop.model.po.EbMerchant;
import com.ishop.model.po.EbProduct;
import com.ishop.model.po.EbProductAttr;
import com.ishop.model.po.EbProductAttrValue;
import com.ishop.model.po.EbProductBrand;
import com.ishop.model.po.EbProductCoupon;
import com.ishop.model.po.EbProductDescription;
import com.ishop.model.vo.ProductAddVo;
import com.ishop.model.vo.ProductCategoryVo;
import com.walker.db.page.GenericPager;
import com.walker.infrastructure.utils.DateUtils;
import com.walker.infrastructure.utils.NumberGenerator;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.web.ResponseValue;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

@RestController
@RequestMapping("/merchant/product")
public class MerchantProductController extends BaseController {

    /**
     * 商品下架
     * @param id
     * @return
     * @date 2023-08-02
     */
    @RequestMapping(value = "/down", method = RequestMethod.POST)
    public ResponseValue productDown(Long id){
        if(id == null || id <= 0){
            return ResponseValue.error(Constants.ERROR_ARGUMENT);
        }
        EbProduct product = this.getProductCache().get(id);
        if(product == null){
            return ResponseValue.error("商品不存在");
        }
        if(product.getIsShow().intValue() == 0){
            return ResponseValue.success("商品已经下架");
        }

        // 设置商品不可见，为更新缓存准备
        product.setIsShow(0);

        this.getProductService().execProductDown(id);
        this.getProductCache().update(product);
        return ResponseValue.success();
    }

    /**
     * 商品上架
     * @param id
     * @return
     * @date 2023-08-02
     */
    @RequestMapping(value = "/up", method = RequestMethod.POST)
    public ResponseValue productUp(Long id){
        if(id == null || id <= 0){
            return ResponseValue.error(Constants.ERROR_ARGUMENT);
        }
        EbProduct product = this.getProductCache().get(id);
        if(product == null){
            return ResponseValue.error("商品不存在");
        }
        if (product.getIsAudit().intValue() == 1) {
            return ResponseValue.error("商品审核中无法上架");
        }
        if (!product.getAuditStatus().equals(ProductConstants.AUDIT_STATUS_EXEMPTION) && !product.getAuditStatus().equals(ProductConstants.AUDIT_STATUS_SUCCESS)) {
            return ResponseValue.error("商品状态异常无法上架");
        }
        if(product.getIsShow().intValue() == 1){
            return ResponseValue.success("商品已经上架");
        }

        int merId = this.getCurrentUser().getMer_id().intValue();
        EbMerchant merchant = this.getMerchantCache().get(merId);
        if (merchant.getIsSwitch().intValue() == 0) {
            return ResponseValue.error("打开商户开关后方能上架商品");
        }

        // 设置商品可见，为更新缓存准备
        product.setIsShow(1);

        // 获取商品skuid
        List<Integer> skuIdList = null;
        List<EbProductAttrValue> productAttrValueList = this.getProductAttrValueService().queryProductAttrValueList(ProductConstants.PRODUCT_TYPE_NORMAL, product.getId());
        if(!StringUtils.isEmptyList(productAttrValueList)){
            skuIdList = new ArrayList<>(productAttrValueList.size());
            for(EbProductAttrValue e : productAttrValueList){
                skuIdList.add(e.getId());
            }
        }
        this.getProductService().execProductUp(product.getId(), skuIdList);
        this.getProductCache().update(product);
        return ResponseValue.success();
    }

    @RequestMapping(value = "/update", method = RequestMethod.POST)
    public ResponseValue update(@RequestBody ProductAddVo request){
        if(request == null || request.getId() == null || request.getId() <= 0){
            return ResponseValue.error(Constants.ERROR_ARGUMENT);
        }
        this.checkProductRequest(request);

        EbProduct product = this.getProductCache().get(request.getId());
        if(product == null){
            return ResponseValue.error("商品不存在");
        }
        if (product.getIsRecycle().intValue() == 1 || product.getIsDel().intValue() == 1) {
            return ResponseValue.error("商品已删除");
        }
        if (product.getIsShow().intValue() == 1) {
            return ResponseValue.error("请先下架商品，再进行修改");
        }
        if (product.getIsAudit().intValue() == 1) {
            return ResponseValue.error("审核中的商品无法修改");
        }

        //
        String cdnUrl = this.getCdnUrl();
        product.setImage(this.clearCdnPrefix(request.getImage()));
        product.setSliderImage(ImageUtils.clearCdnMultiImageUrl(request.getSliderImage(), cdnUrl));
        product.setUpdateTime(DateUtils.getDateTimeNumber());
        product.setName(request.getName());
        product.setUnitName(request.getUnitName());
        product.setIntro(request.getIntro());
        product.setKeyword(request.getKeyword());
        product.setCateId(request.getCateId());
        product.setBrandId(request.getBrandId());
        product.setCategoryId(request.getCategoryId());
        product.setGuaranteeIds(request.getGuaranteeIds());
        product.setTempId(request.getTempId());
        product.setSort(request.getSort());
        product.setSpecType(request.getSpecType()? 1:0);
        product.setIsSub(request.getIsSub()? 1:0);
        product.setAuditStatus(ProductConstants.AUDIT_STATUS_EXEMPTION);
        product.setIsAudit(0);
        // 计算价格
        ProductAttrUtils.combineProductAttrValue(request, product);

        List<EbProductAttr> addAttrList = new ArrayList<>(4);
        List<EbProductAttr> updateAttrList = new ArrayList<>(4);
        for(EbProductAttr e : request.getAttr()){
            if(e.getId() == null || e.getId() <= 0){
                e.setType(ProductConstants.PRODUCT_TYPE_NORMAL);
                e.setProductId(request.getId());
                addAttrList.add(e);
            } else {
                e.setIsDel(0);
                updateAttrList.add(e);
            }
        }

        // 2023-09-10 规格中是否包含价格信息，如果包含说明需要用户选择价格的
//        String[] priceAttr = ProductAttrUtils.hasPriceAttr(request.getAttr());

        List<EbProductAttrValue> addAttrValueList = new ArrayList<>(4);
        List<EbProductAttrValue> updateAttrValueList = new ArrayList<>(4);
        for(EbProductAttrValue attrValue : request.getAttrValue()){
            attrValue.setSku(ProductAttrUtils.getSku(attrValue.getAttrValue()));
            attrValue.setImage(this.clearCdnPrefix(attrValue.getImage()));
            attrValue.setProductId(product.getId());
            attrValue.setVersion(0);
            if(attrValue.getId() == null || attrValue.getId() <= 0){
                // add
                attrValue.setQuota(0);
                attrValue.setQuotaShow(0);
                attrValue.setType(ProductConstants.PRODUCT_TYPE_NORMAL);
                addAttrValueList.add(attrValue);
            } else {
                // update
                attrValue.setIsDel(0);
                updateAttrValueList.add(attrValue);
            }

            // 2023-09-10 规格中是否包含价格信息
//            if(priceAttr != null){
//                double moneyPrice = ProductAttrUtils.acquirePriceAttrMoney(priceAttr, attrValue.getSku());
//                if(moneyPrice != -1){
//                    attrValue.setPrice(moneyPrice);
//                    logger.debug("找到了自定义金额：{}", moneyPrice);
//                }
//            }
        }

        EbProductDescription spd = this.createProductDescription(request.getId(), request.getContent());
        List<EbProductCoupon> couponList = this.createProductCouponList(request.getCouponIds(), request.getId());

        EbMerchant merchant = this.getMerchantCache().get(product.getMerId());
        if (merchant.getProductSwitch().intValue() == 0 && product.getAuditStatus().equals(ProductConstants.AUDIT_STATUS_EXEMPTION)) {
            product.setAuditStatus(ProductConstants.AUDIT_STATUS_EXEMPTION);
        } else {
            product.setAuditStatus(ProductConstants.AUDIT_STATUS_WAIT);
        }

        this.getProductService().execUpdateProduct(product
                , addAttrList, updateAttrList, addAttrValueList, updateAttrValueList, spd, couponList);
        this.getProductCache().update(product);
        this.getProductAttrCache().removeList(ProductAttrUtils.combineKey(ProductConstants.PRODUCT_TYPE_NORMAL, product.getId()));
        logger.info("商户更新商品：" + product.getId());
        return ResponseValue.success();
    }

    @RequestMapping(value = "/save", method = RequestMethod.POST)
    public ResponseValue save(@RequestBody ProductAddVo request){
        if(request == null){
            return ResponseValue.error(Constants.ERROR_ARGUMENT);
        }
//        if (!request.getSpecType() && request.getAttrValue().size() > 1) {
//            throw new PlatformRuntimeException("单规格商品属性值不能大于1");
//        }
//        if (request.getIsSub()) {
//            request.getAttrValue().forEach(av -> {
//                double brokerageRatio = av.getBrokerage() + av.getBrokerageTwo();
//                if (brokerageRatio > VariableConstants.retailStoreBrokerageRatio) {
//                    throw new PlatformRuntimeException("一二级返佣比例之和范围为 0~{}" + VariableConstants.retailStoreBrokerageRatio);
//                }
//            });
//        }
        this.checkProductRequest(request);

        int merchantId = this.getCurrentUser().getMer_id().intValue();

        EbProduct product = VoUtils.acquireProduct(request, merchantId);
        product.setImage(this.clearCdnPrefix(request.getImage()));
        if(StringUtils.isNotEmpty(request.getSliderImage())){
            // 2023-08-01 多个图片，应该先分别裁剪前缀。
//            product.setSliderImage(this.clearCdnPrefix(request.getSliderImage()));
            product.setSliderImage(ImageUtils.clearCdnMultiImageUrl(request.getSliderImage(), this.getCdnUrl()));
        }
        if(StringUtils.isNotEmpty(request.getFlatPattern())){
            product.setFlatPattern(this.clearCdnPrefix(request.getFlatPattern()));
        }

        long productId = NumberGenerator.getLongSequenceNumber();

        //计算价格
//        List<EbProductAttrValue> attrValueAddRequestList = request.getAttrValue();
//        EbProductAttrValue minAttrValue = attrValueAddRequestList.stream().min(Comparator.comparing(EbProductAttrValue::getPrice)).get();
//        product.setPrice(minAttrValue.getPrice());
//        product.setOtPrice(minAttrValue.getOtPrice());
//        product.setCost(minAttrValue.getCost());
//        product.setStock(attrValueAddRequestList.stream().mapToInt(EbProductAttrValue::getStock).sum());
        ProductAttrUtils.combineProductAttrValue(request, product);
        product.setCreateTime(DateUtils.getDateTimeNumber());
        product.setUpdateTime(product.getCreateTime());
        product.setId(productId);

        List<EbProductAttr> addRequestList = request.getAttr();
        List<EbProductAttr> attrList = addRequestList.stream().map(e -> {
//            e.setId(id);
            e.setType(ProductConstants.PRODUCT_TYPE_NORMAL);
            e.setProductId(productId);
            return e;
        }).collect(Collectors.toList());

//        // 2023-09-10 规格中是否包含价格信息，如果包含说明需要用户选择价格的
//        String[] priceAttr = ProductAttrUtils.hasPriceAttr(attrList);

        for(EbProductAttrValue attrValue : request.getAttrValue()){
//            attrValue.setId(id);
            attrValue.setSku(ProductAttrUtils.getSku(attrValue.getAttrValue()));
            attrValue.setQuota(0);
            attrValue.setQuotaShow(0);
            attrValue.setType(ProductConstants.PRODUCT_TYPE_NORMAL);
            attrValue.setImage(this.clearCdnPrefix(attrValue.getImage()));
            attrValue.setProductId(productId);

            // 2023-09-10 规格中是否包含价格信息
//            if(priceAttr != null){
//                double moneyPrice = ProductAttrUtils.acquirePriceAttrMoney(priceAttr, attrValue.getSku());
//                if(moneyPrice != -1){
//                    attrValue.setPrice(moneyPrice);
//                    logger.debug("找到了自定义金额：{}", moneyPrice);
//                }
//            }
        }

        EbProductDescription spd = this.createProductDescription(productId, request.getContent());

        List<EbProductCoupon> couponList = null;
        if(!StringUtils.isEmptyList(request.getCouponIds())){
            couponList = this.createProductCouponList(request.getCouponIds(), productId);
        }

        this.getProductService().execInsertProduct(product, attrList, request.getAttrValue(), spd, couponList);
        // 2023-08-01 加入商品缓存
        this.getProductCache().save(product);
        logger.info("商户新增商品：" + productId);
        return ResponseValue.success();
    }

    private List<EbProductCoupon> createProductCouponList(List<Integer> couponIds, long productId){
        List<EbProductCoupon> couponList = null;
        if(!StringUtils.isEmptyList(couponIds)){
            couponList = new ArrayList<>(couponIds.size());
            EbProductCoupon productCoupon = null;
            for (Integer couponId : couponIds) {
                productCoupon = new EbProductCoupon();
                productCoupon.setId(NumberGenerator.getLongSequenceNumber());
                productCoupon.setProductId(productId);
                productCoupon.setCouponId(couponId);
                productCoupon.setAddTime(VariableConstants.getNowTime());
                couponList.add(productCoupon);
            }
        }
        return couponList;
    }

    private EbProductDescription createProductDescription(long productId, String content){
        EbProductDescription spd = new EbProductDescription();
        spd.setDescription(StringUtils.isNotEmpty(content) ? this.clearCdnPrefix(content) : StringUtils.EMPTY_STRING);
        spd.setType(ProductConstants.PRODUCT_TYPE_NORMAL);
        spd.setProductId(productId);
        spd.setId(NumberGenerator.getLongSequenceNumber());
        return spd;
    }

    private void checkProductRequest(ProductAddVo request){
        if (!request.getSpecType() && request.getAttrValue().size() > 1) {
            throw new PlatformRuntimeException("单规格商品属性值不能大于1");
        }
        if (request.getIsSub()) {
            request.getAttrValue().forEach(av -> {
                double brokerageRatio = av.getBrokerage() + av.getBrokerageTwo();
                if (brokerageRatio > VariableConstants.retailStoreBrokerageRatio) {
                    throw new PlatformRuntimeException("一二级返佣比例之和范围为 0~{}" + VariableConstants.retailStoreBrokerageRatio);
                }
            });
        }
    }

    /**
     * 拷贝（采集）商品方法。可以是第三方各种平台。
     * @param request
     * @return
     * @date 2023-06-21
     */
    @RequestMapping(value = "/copy/product", method = RequestMethod.POST)
    public ResponseValue copyProduct(@RequestBody CopyProductRequest request){

        return ResponseValue.success();
    }

    /**
     * 商户端关联展示品牌分类关联数据。
     * @param param
     * @return
     * @date 2023-06-15
     */
    @RequestMapping(value = "/brand/list", method = RequestMethod.GET)
    public ResponseValue pageBrandList(BrandCategoryParam param){
        if(param == null || param.getCid() == null){
            return ResponseValue.error(Constants.ERROR_ARGUMENT);
        }
        GenericPager<EbProductBrand> pager = this.getProductBrandService().queryPageBrandListByCategory(param.getCid(), param.getBrandName());
        return ResponseValue.success(pager);
    }

    @RequestMapping(value = "/list", method = RequestMethod.GET)
    public ResponseValue list(ProductParam productParam){
        long merchantId = this.getCurrentUser().getMer_id();
        if(merchantId <= 0){
            return ResponseValue.error("您不是商户，无法使用该功能");
        }
        productParam.setMerId((int)merchantId);
        GenericPager<EbProduct> pager = this.getProductService().queryPageMerchantProductList(productParam);
        if(!StringUtils.isEmptyList(pager.getDatas())){
            for(EbProduct e : pager.getDatas()){
                e.setParameterInt("collectCount", 0);
                if(StringUtils.isNotEmpty(e.getImage())){
                    e.setImage(this.getCdnUrl() + e.getImage());
                }
            }
        }
        return ResponseValue.success(pager);
    }

    @RequestMapping(value = "/tabs/headers", method = RequestMethod.GET)
    public ResponseValue getTabsHeader(){
        logger.warn("暂时给出商品测试标签页");
        return ResponseValue.success(this.getProductService().queryTabsHeader());
    }

    @RequestMapping(value = "/cache/tree", method = RequestMethod.GET)
    public ResponseValue getMerchantCategoryCacheTree(){
        List<ProductCategoryVo> list = this.getProductCategoryCache().getTree(1);
        return ResponseValue.success(list);
    }
}
