package com.ishop.merchant.platform;

import com.iplatform.base.Constants;
import com.iplatform.base.util.MenuUtils;
import com.iplatform.model.to.UserAndDeptTo;
import com.ishop.merchant.BaseController;
import com.ishop.merchant.pojo.MerchantParam;
import com.ishop.merchant.util.ImageUtils;
import com.ishop.merchant.util.VoUtils;
import com.ishop.model.po.EbMerchant;
import com.ishop.model.po.EbMerchantInfo;
import com.ishop.model.vo.PlatformMerDetailVo;
import com.walker.db.page.GenericPager;
import com.walker.infrastructure.time.TimeRange;
import com.walker.infrastructure.time.TimeRangeUtils;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.web.OrgType;
import com.walker.web.ResponseValue;
import com.walker.web.util.IdUtils;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

@RestController
@RequestMapping("/platform/merchant")
public class MerchantController extends BaseController {

    /**
     * 平台编辑商户，仅能该基本信息。其他要商户自己维护。
     * @return
     * @date 2023-07-20
     */
    @RequestMapping(value = "/update", method = RequestMethod.POST)
    public ResponseValue update(@RequestBody EbMerchant merchant){
        if(merchant == null || merchant.getId() == null || merchant.getId() <= 0){
            return ResponseValue.error("编辑的商户不存在");
        }

        String qualificationPicture = merchant.getQualificationPicture();
        if(StringUtils.isNotEmpty(qualificationPicture)){
            merchant.setQualificationPicture(ImageUtils.clearCdnMultiImageUrl(qualificationPicture, this.getCdnUrl()));
        }
        if(StringUtils.isEmpty(merchant.getRemark())){
            merchant.setRemark(merchant.getKeywords());
        }

        UserAndDeptTo userAndDeptTo = new UserAndDeptTo();
        userAndDeptTo.setDeptName(merchant.getName());
        userAndDeptTo.setNickName(merchant.getName());
        userAndDeptTo.setPhone(merchant.getPhone());
        userAndDeptTo.setMail(merchant.getEmail());
        userAndDeptTo.setOrgId(merchant.getId());

        this.getMerchantService().execUpdateMerchantPlatform(merchant, userAndDeptTo);
        this.getMerchantCache().update(merchant);
        logger.info("平台更新商户基本信息成功：{}", merchant.getName());
        return ResponseValue.success();
    }

    /**
     * 平台查看商户详情。
     * @param id
     * @return
     * @date 2023-07-20
     */
    @RequestMapping(value = "/detail", method = RequestMethod.GET)
    public ResponseValue info(Integer id){
        if(id == null || id <= 0){
            return ResponseValue.error("商户编号错误");
        }
        EbMerchant merchant = this.getMerchantCache().get(id);
        PlatformMerDetailVo vo = VoUtils.acquirePlatformMerDetailVo(merchant);
        if(StringUtils.isNotEmpty(merchant.getQualificationPicture())){
            String imageList = ImageUtils.combineMultiImageUrl(merchant.getQualificationPicture(), this.getCdnUrl());
            vo.setQualificationPicture(imageList);
        }
        return ResponseValue.success(vo);
    }

    @RequestMapping(value = "/list", method = RequestMethod.GET)
    public ResponseValue list(MerchantParam merchantParam){
        if(merchantParam == null){
            return ResponseValue.error(Constants.ERROR_ARGUMENT);
        }
        logger.debug(merchantParam.toString());
        // 时间范围选择
        TimeRange timeRange = TimeRangeUtils.getDateLimit(merchantParam.getDateLimit());
        // 分页查询列表
        GenericPager<EbMerchant> pager = this.getMerchantService().queryPageMerchantList(merchantParam.getCategoryId()
                , merchantParam.getTypeId(), merchantParam.getPhone(), merchantParam.getIsSelf(), merchantParam.getIsSwitch()
                , timeRange, merchantParam.getKeywords());
        return ResponseValue.success(pager);
    }

    @RequestMapping(value = "/header/num", method = RequestMethod.GET)
    public ResponseValue listHeaderNum(MerchantParam merchantParam){
        if(merchantParam == null){
            return ResponseValue.error(Constants.ERROR_ARGUMENT);
        }
        logger.debug("date = " + merchantParam.getDateLimit());
        TimeRange timeRange = TimeRangeUtils.getDateLimit(merchantParam.getDateLimit());
        int openCount = this.getMerchantService().countMerchantTotal(merchantParam.getCategoryId()
                , merchantParam.getTypeId(), merchantParam.getPhone(), merchantParam.getIsSelf(), true, timeRange, merchantParam.getKeywords());
        int closeCount = this.getMerchantService().countMerchantTotal(merchantParam.getCategoryId()
                , merchantParam.getTypeId(), merchantParam.getPhone(), merchantParam.getIsSelf(), false, timeRange, merchantParam.getKeywords());

        Map<String, Object> data = new HashMap<>(4);
        data.put("openNum", openCount);
        data.put("closeNum", closeCount);
        return ResponseValue.success(data);
    }

    @RequestMapping(value = "/add", method = RequestMethod.POST)
    public ResponseValue add(@RequestBody EbMerchant ebMerchant){
        if(ebMerchant == null
                || StringUtils.isEmpty(ebMerchant.getName())
                || StringUtils.isEmpty(ebMerchant.getPhone())){
            return ResponseValue.error(Constants.ERROR_ARGUMENT);
        }
        String name = ebMerchant.getName().trim().toLowerCase();
        String phone = ebMerchant.getPhone().trim().toLowerCase();

        if(this.getMerchantApplyService().queryApplyByName(name) != null){
            return ResponseValue.error("相同商户名称已在审核中");
        }
        if(this.getMerchantApplyService().queryApplyByPhone(phone) != null){
            return ResponseValue.error("相同商户手机号已在审核中");
        }

        if(this.getMerchantService().queryMerchantByName(name) != null){
            return ResponseValue.error("商户名称已存在");
        }
        if(this.getMerchantService().queryMerchantByPhone(phone) != null){
            return ResponseValue.error("商户手机号已注册");
        }

        String qualificationPicture = ebMerchant.getQualificationPicture();
        if(StringUtils.isNotEmpty(qualificationPicture)){
            ebMerchant.setQualificationPicture(this.clearCdnPrefix(qualificationPicture));
        }
        if(StringUtils.isEmpty(ebMerchant.getRemark())){
            ebMerchant.setRemark(ebMerchant.getKeywords());
        }
        // 2023-06-08 设置商户唯一标识uuid，外部使用
        ebMerchant.setUuid(IdUtils.randomUUID());

        //
        long loginId = this.getCurrentUserId();
        UserAndDeptTo userAndDeptTo = new UserAndDeptTo();
        userAndDeptTo.setCreateId(String.valueOf(loginId));
        userAndDeptTo.setDeptName(ebMerchant.getName());
        userAndDeptTo.setOrgType(OrgType.TYPE_ORG);
        userAndDeptTo.setUserName(ebMerchant.getPhone());
        userAndDeptTo.setNickName(ebMerchant.getRealName());
        userAndDeptTo.setPhone(ebMerchant.getPhone());
        userAndDeptTo.setMail(ebMerchant.getEmail());
        userAndDeptTo.setAvatar(ebMerchant.getAvatar());
        userAndDeptTo.setMenuType(MenuUtils.MENU_SCOPE_ORG);
        userAndDeptTo.setPassword(this.encryptPassword(com.ishop.merchant.Constants.INIT_PASSWORD));

        this.getMerchantService().execInsertMerchant(ebMerchant, com.ishop.merchant.Constants.CREATE_TYPE_ADMIN, loginId, userAndDeptTo);
        logger.info("后台创建商户成功：{}", ebMerchant);
        return ResponseValue.success();
    }

    @RequestMapping(value = "/close", method = RequestMethod.POST)
    public ResponseValue close(Integer id){

        return ResponseValue.success();
    }

    @RequestMapping(value = "/open", method = RequestMethod.POST)
    public ResponseValue open(Integer id){
        EbMerchant merchant = this.getMerchantService().get(new EbMerchant(id));
        if(merchant.getIsSwitch() == 1){
            return ResponseValue.error("商户已开启");
        }
        if (StringUtils.isEmpty(merchant.getAvatar())
                || StringUtils.isEmpty(merchant.getBackImage())
                || StringUtils.isEmpty(merchant.getStreetBackImage())) {
            return ResponseValue.error("请先进行商户头像、背景图配置");
        }

        EbMerchantInfo queryInfo = new EbMerchantInfo();
        queryInfo.setMerId(id);
        List<EbMerchantInfo> merchantInfoList = this.getMerchantService().select(queryInfo);
        if(StringUtils.isEmptyList(merchantInfoList)){
            return ResponseValue.error("商户未找到配置记录");
        }

        EbMerchantInfo merchantInfo = merchantInfoList.get(0);
        if (StringUtils.isEmpty(merchantInfo.getServiceLink()) && StringUtils.isEmpty(merchantInfo.getServicePhone())) {
            return ResponseValue.error("请先进行客服信息配置");
        }

        this.getMerchantService().execOpenMerchant(id);
        return ResponseValue.success();
    }
}
