package com.ishop.merchant.platform;

import com.iplatform.base.Constants;
import com.iplatform.base.service.RoleServiceImpl;
import com.iplatform.base.util.TextUtils;
import com.iplatform.core.BeanContextAware;
import com.iplatform.model.po.S_role;
import com.iplatform.model.po.S_user_core;
import com.iplatform.model.po.S_user_login;
import com.ishop.merchant.BaseController;
import com.ishop.merchant.pojo.UserSearchParam;
import com.ishop.model.po.EbUser;
import com.ishop.model.request.UserAssignTagRequest;
import com.ishop.model.request.UserBalanceRequest;
import com.ishop.model.vo.RoleMerchantVo;
import com.walker.db.page.GenericPager;
import com.walker.infrastructure.utils.KeyValue;
import com.walker.infrastructure.utils.PhoneNumberUtils;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.web.ResponseValue;
import com.walker.web.UserOnlineProvider;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import java.util.ArrayList;
import java.util.List;

@RestController
@RequestMapping("/platform/user")
public class UserRegisterController extends BaseController {

    @RequestMapping(value = "/tag", method = RequestMethod.POST)
    public ResponseValue setUserTag(@RequestBody UserAssignTagRequest request){
        if(request == null
            || StringUtils.isEmpty(request.getTagIds())
            || StringUtils.isEmpty(request.getIds())){
            return ResponseValue.error("设置标签参数未提交");
        }
        List<Long> userIds = TextUtils.stringToLongArrayByRegex(request.getIds(), StringUtils.DEFAULT_SPLIT_SEPARATOR);
        // 更新用户标签
        this.getUserRegisterService().execBatchTag(userIds, request.getTagIds());
        // 更新用户缓存
        this.batchUpdateEbUserCache(userIds);
        return ResponseValue.success();
    }

    /**
     * 保存移动端，作为商户管理角色设置。
     * @param request
     * @return
     * @date 2023-09-25
     */
    @RequestMapping(value = "/roles/merchant/save", method = RequestMethod.POST)
    public ResponseValue saveRegUserMerchantAdmin(@RequestBody RoleMerchantVo request){
        if(request == null
            || StringUtils.isEmpty(request.getRoleId())
            || request.getMerId() <= 0){
            return ResponseValue.error("请设置管理角色信息");
        }
        if(request.getUserId() <= 0){
            return ResponseValue.error("用户信息错误，userId为空");
        }
        long roleId = Long.parseLong(request.getRoleId());

        this.getUserRegisterService().execRoleMerchantAdmin(request.getType(), roleId, request.getMerId(), request.getUserId());
        // 1.更新用户缓存，2.删除用户登录缓存
        S_user_core userCore = this.getUser(request.getUserId());
        userCore.setType(request.getType());
        userCore.setMer_id(request.getMerId());
        this.getUserCacheProvider().updateUser(userCore);

        S_user_login user_login = this.getUserLoginCache().getUserLogin(userCore.getUser_name());
        if(user_login == null){
            logger.warn("移动端商户角色设置：登录缓存未查到用户信息，无法删除登录缓存。user = {}", userCore.getUser_name());
            return ResponseValue.success();
        }
        String uuid = user_login.getUuid();
        UserOnlineProvider userOnlineProvider = BeanContextAware.getBeanByType(UserOnlineProvider.class);
        userOnlineProvider.removeUserPrincipal(uuid);
        logger.debug("删除用户登录缓存：uuid={}", uuid);
        return ResponseValue.success();
    }

    @RequestMapping(value = "/roles/merchant", method = RequestMethod.GET)
    public ResponseValue showUserMerchantRole(Long id){
        if(id == null || id <= 0){
            return ResponseValue.error("id is required!");
        }
        List<S_role> roleList = BeanContextAware.getBeanByType(RoleServiceImpl.class).queryUserRoleList(id);
        S_user_core user = this.getUser(id);

        RoleMerchantVo vo = new RoleMerchantVo();
        vo.setType(user.getType());
        vo.setMerId(user.getMer_id());
        vo.setUserId(id);
        if(!StringUtils.isEmptyList(roleList)){
            // 移动端只会配置一个角色。
            vo.setRoleId(String.valueOf(roleList.get(0).getRole_id()));
        } else {
            vo.setRoleId(StringUtils.EMPTY_STRING);
        }
        return ResponseValue.success(vo);
    }

    /**
     * 在平台端，管理用户，可以设置移动端用户为商户管理员，看到一些管理功能界面。
     * <pre>
     *     1) 设置移动端用户属于某个角色，如：明运管理员等
     *     2)
     * </pre>
     * @return
     * @date 2023-09-25
     */
    @RequestMapping(value = "/roles", method = RequestMethod.GET)
    public ResponseValue listRole(){
        List<S_role> roleList = this.getUserService().select(new S_role());
        List<KeyValue> data = new ArrayList<>(8);
        if(!StringUtils.isEmptyList(roleList)){
            for(S_role role : roleList){
                data.add(new KeyValue(role.getRole_id().toString(), role.getRole_name()));
            }
        }
        return ResponseValue.success(data);
    }

    @RequestMapping(value = "/list", method = RequestMethod.GET)
    public ResponseValue list(UserSearchParam param){
//        if(param == null){
//            return ResponseValue.error(Constants.ERROR_ARGUMENT);
//        }
        GenericPager<EbUser> pager = this.getUserRegisterService().queryPageUserList(param);
        if(!StringUtils.isEmptyList(pager.getDatas())){
            for(EbUser e : pager.getDatas()){
                if(StringUtils.isNotEmpty(e.getAvatar())){
                    e.setAvatar(this.getCdnUrl() + e.getAvatar());
                }
            }
        }
        return ResponseValue.success(pager);
    }

    @RequestMapping(value = "/detail", method = RequestMethod.GET)
    public ResponseValue detail(Long id){
        if(id == null || id.longValue() <= 0){
            return ResponseValue.error(Constants.ERROR_ARGUMENT);
        }
        EbUser user = this.getUserRegCache().get(id);
        if(user == null){
            return ResponseValue.error("无效用户");
        }
        if(user.getSpreadUid() != null && user.getSpreadUid().longValue() > 0){
            user.setParameterString("spreadName", this.getUserRegisterName(user.getSpreadUid()));
        }
        // 手机号脱敏，放在另外属性中
        if(StringUtils.isNotEmpty(user.getPhone())){
            user.setParameterString("userPhone", PhoneNumberUtils.maskMobile(user.getPhone()));
        }
        return ResponseValue.success(user);
    }

    /**
     * 用户后台充值，变动余额。
     * @param request
     * @return
     * @date 2023-09-06
     */
    @RequestMapping(value = "/operate/balance", method = RequestMethod.GET)
    public  ResponseValue userBalance(UserBalanceRequest request){
        if(request == null || request.getUid() == null){
            return ResponseValue.error(Constants.ERROR_ARGUMENT);
        }
        if(request.getMoney() == null || request.getMoney().doubleValue() == 0){
            return ResponseValue.error("充值金额为0");
        }
        EbUser user = this.getUserRegCache().get(request.getUid());
        if(user == null){
            return ResponseValue.error("用户不存在");
        }
        if (user.getIsLogoff().intValue() == 1) {
            return ResponseValue.error("已注销用户不可操作");
        }

        // 减少时要判断小于0的情况,添加时判断是否超过数据限制
        if (request.getOperateType().equals(com.ishop.merchant.Constants.OPERATION_TYPE_SUBTRACT)) {
            if ((user.getNowMoney().doubleValue() - request.getMoney().doubleValue()) < 0) {
                return ResponseValue.error("余额扣减后不能小于0");
            }
        } else if(request.getOperateType().equals(com.ishop.merchant.Constants.OPERATION_TYPE_ADD)){
            if((user.getNowMoney().doubleValue() + request.getMoney().doubleValue()) > 99999999.99){
                return ResponseValue.error("余额添加后后不能大于99999999.99");
            }
        } else {
            throw new UnsupportedOperationException("不支持的操作类型：" + request.getOperateType());
        }

        this.getUserRegisterService().execModifyUserBalance(request.getUid(), request.getMoney().doubleValue(), request.getOperateType(), user.getNowMoney().doubleValue());
        return ResponseValue.success();
    }
}
