package com.ishop.merchant.service;

import com.ishop.merchant.CouponConstants;
import com.ishop.merchant.util.VoUtils;
import com.ishop.model.po.EbCoupon;
import com.ishop.model.po.EbCouponUser;
import com.ishop.model.vo.CouponFrontVo;
import com.walker.db.page.GenericPager;
import com.walker.db.page.ListPageContext;
import com.walker.db.page.PageSearch;
import com.walker.infrastructure.utils.DateUtils;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.jdbc.service.BaseServiceImpl;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Service
public class CouponServiceImpl extends BaseServiceImpl {

    private CouponUserServiceImpl couponUserService;

    @Autowired
    public CouponServiceImpl(CouponUserServiceImpl couponUserService){
        this.couponUserService = couponUserService;
    }

    /**
     * 移动端优惠券列表
     * @param userId 当前用户ID，如果0表示无用户
     * @param category 分类
     * @param merId 商户ID
     * @param productId 商品ID
     * @return
     * @date 2023-07-09
     */
    public GenericPager<CouponFrontVo> queryPageH5CouponList(long userId, Integer category, Integer merId, Long productId){
        if((merId == null || merId.intValue() <=0) && (productId == null || productId.longValue() <= 0)){
            throw new IllegalArgumentException("商户ID与商品ID不能都为空");
        }

        List<CouponFrontVo> couponFrontVoList = new ArrayList<>();

        PageSearch pageSearch = ListPageContext.getPageSearch();
        GenericPager<EbCoupon> pager = this.queryH5CouponList(category, merId, productId, pageSearch);
        List<EbCoupon> list = pager.getDatas();
        if(StringUtils.isEmptyList(list)){
            return ListPageContext.createGenericPager(couponFrontVoList, 1, 0);
        }

        //获取用户当前已领取的优惠券
        Map<Integer, EbCouponUser> couponUserMap = this.couponUserService.queryCouponIdAndValue(userId);

        CouponFrontVo couponFrontVo = null;
        for (EbCoupon coupon : list) {
            couponFrontVo = VoUtils.acquireCouponFrontVo(coupon);
            if(userId > 0 && couponUserMap != null && couponUserMap.containsKey(coupon.getId())){
                couponFrontVo.setIsUse(true);
            }
            // 更改使用时间格式，去掉时分秒
            couponFrontVo.setUseStartTimeStr(DateUtils.toShowDate((int)(coupon.getUseStartTime()/1000000)));
            couponFrontVo.setUseEndTimeStr(DateUtils.toShowDate((int)(coupon.getUseEndTime()/1000000)));
            couponFrontVoList.add(couponFrontVo);
        }
        return ListPageContext.createGenericPager(couponFrontVoList, pager.getPageIndex(), pager.getPageSize());
    }

    /**
     * 根据条件查询优惠券集合，前端使用。
     * @param category
     * @param merId
     * @param productId
     * @return
     * @date 2023-07-09
     */
    private GenericPager<EbCoupon> queryH5CouponList(Integer category, Integer merId, Long productId, PageSearch pageSearch){
        Map<String, Object> param = new HashMap<>(4);
        StringBuilder sql = new StringBuilder("select c.* from eb_coupon as c where c.is_del = 0 and c.status = 1 and receive_type = 1");
        sql.append(" and (last_total > 0 or is_limited = 0)");
        sql.append(" and (is_time_receive = 0 or (receive_start_time <= :date and receive_end_time >= :date))");
        sql.append(" and (is_fixed_time = 0 or use_end_time >= :date)");
        param.put("date", DateUtils.getDateTimeNumber());   // 格式如：20230709204010
        if(category != null){
            sql.append(" and category = :category");
            param.put("category", category);
        }
        if(merId != null){
            sql.append(" and mer_id = :merId");
            param.put("merId", merId);
        }
        if(productId != null){
            sql.append(" and :productId in (select pid from eb_coupon_product where cid = c.id)");
            param.put("productId", productId);
        }
        sql.append(" order by c.sort,c.id desc");
        return this.selectSplit(sql.toString(), param, pageSearch.getPageIndex(), pageSearch.getPageSize(), new EbCoupon());
    }

    /**
     * 商品可用优惠券列表（商品创建时选择使用）
     * @param merId
     * @return
     * @date 2023-06-21
     */
    public List<EbCoupon> queryProductUsableList(int merId){
        EbCoupon coupon = new EbCoupon();
        coupon.setIsDel(0);
        coupon.setMerId(merId);
        coupon.setReceiveType(CouponConstants.COUPON_RECEIVE_TYPE_PAY_PRODUCT);
        coupon.setStatus(1);
        return this.select(coupon);
    }
}
