package com.ishop.merchant.service;

import com.iplatform.base.service_api.UserAndDeptServiceApi;
import com.iplatform.model.to.UserAndDeptTo;
import com.iplatform.model.to.UserAndDeptToResult;
import com.ishop.merchant.Constants;
import com.ishop.model.po.EbMerchant;
import com.ishop.model.po.EbMerchantDailyStatement;
import com.ishop.model.po.EbMerchantInfo;
import com.ishop.model.po.EbMerchantMonthStatement;
import com.ishop.model.po.EbShippingTemplates;
import com.walker.db.page.GenericPager;
import com.walker.infrastructure.time.TimeRange;
import com.walker.infrastructure.utils.DateUtils;
import com.walker.infrastructure.utils.NumberGenerator;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.jdbc.service.BaseServiceImpl;
import com.walker.web.ResponseValue;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Service
public class MerchantServiceImpl extends BaseServiceImpl {

    private UserAndDeptServiceApi userAndDeptServiceApi;

    @Autowired
    public MerchantServiceImpl(UserAndDeptServiceApi userAndDeptServiceApi){
        this.userAndDeptServiceApi = userAndDeptServiceApi;
    }

    public void execOpenMerchant(int id){
        EbMerchant merchant = new EbMerchant();
        merchant.setId(id);
        merchant.setIsSwitch(1);

        this.save(merchant);
    }

    /**
     * 更新商户配置信息。
     * @param merchant
     * @param merchantInfo
     * @date 2023-06-09
     */
    public void execUpdateConfigInfo(EbMerchant merchant, EbMerchantInfo merchantInfo){
        this.save(merchant);
        this.save(merchantInfo);
    }

    /**
     * 平台更新商户基本信息，同步到系统机构。
     * @param merchant
     * @param userAndDeptTo
     * @date 2023--7-20
     */
    public void execUpdateMerchantPlatform(EbMerchant merchant, UserAndDeptTo userAndDeptTo){
        ResponseValue responseValue = this.userAndDeptServiceApi.execUpdateTopOrgAndAdmin(userAndDeptTo);
        if(!responseValue.isState()){
            throw new IllegalStateException("商户机构更新错误：" + responseValue.getMsg());
        }
        this.save(merchant);
    }

    /**
     * 创建新商户。
     * @param merchant
     * @param createType
     * @param createId
     * @param userAndDeptTo
     */
    public void execInsertMerchant(EbMerchant merchant, String createType, long createId, UserAndDeptTo userAndDeptTo){
        // 初始化管理员账号
        // 初始化系统顶级机构
        ResponseValue responseValue = this.userAndDeptServiceApi.execInsertTopOrgAndAdmin(userAndDeptTo);
        if(!responseValue.isState()){
            throw new IllegalStateException("商户机构创建错误：" + responseValue.getMsg());
        }

        UserAndDeptToResult userAndDeptToResult = (UserAndDeptToResult) responseValue.getData();

//        int merchantId = this.queryNextId();
        // 机构生成为int，强制转换即可
        int merchantId = (int)userAndDeptToResult.getDeptId();
        int otherId = Integer.parseInt(new StringBuilder(merchantId).append(1).toString());
        merchant.setId(merchantId);
        merchant.setCreateId(createId);
        merchant.setCreateType(createType);
        merchant.setCreateTime(DateUtils.getDateTimeNumber());
        merchant.setUpdateTime(merchant.getCreateTime());
        merchant.setAdminId(userAndDeptToResult.getUserId());
        this.insert(merchant);

        EbMerchantInfo merchantInfo = new EbMerchantInfo();
        merchantInfo.setId(merchantId);
        merchantInfo.setMerId(merchantId);
        this.insert(merchantInfo);

        // 初始化日/月帐单
        EbMerchantDailyStatement dailyStatement = new EbMerchantDailyStatement();
        EbMerchantMonthStatement monthStatement = new EbMerchantMonthStatement();
        dailyStatement.setId(NumberGenerator.getLongSequenceNumber());
        dailyStatement.setMerId(merchantId);
        dailyStatement.setDataDate(DateUtils.getDateForHuman(System.currentTimeMillis()));
        monthStatement.setId(dailyStatement.getId());
        monthStatement.setMerId(merchantId);
        monthStatement.setDataDate(dailyStatement.getDataDate().substring(0, 7));
        this.insert(dailyStatement);
        this.insert(monthStatement);

        // 初始化一条全国包邮的运费模板
        EbShippingTemplates shippingTemplates = new EbShippingTemplates();
        shippingTemplates.setId(otherId);
        shippingTemplates.setName(Constants.DEFAULT_NAME);
        shippingTemplates.setType(Constants.CHARGE_MODE_TYPE_UNKNOWN);
        shippingTemplates.setAppoint(Constants.APPOINT_TYPE_ALL);
        shippingTemplates.setSort(999);
        shippingTemplates.setMerId(merchantId);
        this.insert(shippingTemplates);
    }

    /**
     * 返回商户配置扩展信息。
     * @param merId
     * @return
     * @date 2023-07-04
     */
    public EbMerchantInfo queryMerchantInfo(int merId){
        EbMerchantInfo info = new EbMerchantInfo();
        info.setMerId(merId);
        List<EbMerchantInfo> list = this.select(info);
        return list.get(0);
    }

    public EbMerchant queryMerchantByName(String name){
        EbMerchant ebMerchant = new EbMerchant();
        ebMerchant.setName(name);
        ebMerchant.setIsDel(0);
        List<EbMerchant> list = this.select(ebMerchant);
        if(StringUtils.isEmptyList(list)){
            return null;
        }
        return list.get(0);
    }
    public EbMerchant queryMerchantByPhone(String phone){
        EbMerchant ebMerchant = new EbMerchant();
        ebMerchant.setPhone(phone);
        ebMerchant.setIsDel(0);
        List<EbMerchant> list = this.select(ebMerchant);
        if(StringUtils.isEmptyList(list)){
            return null;
        }
        return list.get(0);
    }

    public GenericPager<EbMerchant> queryPageMerchantList(Integer categoryId
            , Integer typeId, String phone, Boolean isSelf, Boolean isSwitch, TimeRange timeRange, String keywords){
        Object[] sqlAndParameter = this.combineSqlAndParameter(SQL_QUERY_PREFIX_LIST
                , categoryId, typeId, phone, isSelf, isSwitch, timeRange, keywords);
        return this.selectSplit(sqlAndParameter[0].toString(), (Map<String, Object>)sqlAndParameter[1], new EbMerchant());
    }

    public int countMerchantTotal(Integer categoryId
            , Integer typeId, String phone, Boolean isSelf, Boolean isSwitch, TimeRange timeRange, String keywords){
        Object[] sqlAndParameter = this.combineSqlAndParameter(SQL_QUERY_PREFIX_COUNT
                , categoryId, typeId, phone, isSelf, isSwitch, timeRange, keywords);
        return this.queryForInt(sqlAndParameter[0].toString(), (Map<String, Object>)sqlAndParameter[1]);
    }

    private Object[] combineSqlAndParameter(String sqlPrefix, Integer categoryId
            , Integer typeId, String phone, Boolean isSelf, Boolean isSwitch, TimeRange timeRange, String keywords){
        Map<String, Object> parameters = new HashMap<>(4);
        StringBuilder sql = new StringBuilder(sqlPrefix);
        if(categoryId != null && categoryId.intValue() > 0){
            sql.append(" and category_id=:categoryId");
            parameters.put("categoryId", categoryId);
        }
        if(typeId != null && typeId.intValue() >= 0){
            sql.append(" and type_id=:typeId");
            parameters.put("typeId", typeId);
        }
        if(StringUtils.isNotEmpty(phone)){
            sql.append(" and phone=:phone");
            parameters.put("phone", phone);
        }
        if(isSelf != null){
            sql.append(" and is_self=:isSelf");
            parameters.put("isSelf", isSelf? 1:0);
        }
        if(isSwitch != null){
            sql.append(" and is_switch=:isSwitch");
            parameters.put("isSwitch", isSwitch? 1:0);
        }
        if(timeRange != null){
            if(timeRange.getStartTime() != null){
                sql.append(" and create_time >= :startTime");
                parameters.put("startTime", timeRange.getStartTime());
            }
            if(timeRange.getEndTime() != null){
                sql.append(" and create_time <= :endTime");
                parameters.put("endTime", timeRange.getEndTime());
            }
        }
        if(StringUtils.isNotEmpty(keywords)){
            sql.append(" and (name like :k1 or keywords like :k2 or remark like :k3)");
            String likeParam = "%" + keywords + "%";
            parameters.put("k1", likeParam);
            parameters.put("k2", likeParam);
            parameters.put("k3", likeParam);
        }
        Object[] result = new Object[2];
        result[0] = sql.toString();
        result[1] = parameters;
        return result;
    }

    /**
     * 获得下一个可用的最大ID。
     * @return
     * @date 2023-05-17
     */
    public int queryNextId(){
        int maxId = this.queryForInt(SQL_MAX_ID, new Object[]{});
        return maxId+1;
    }

    private static final String SQL_MAX_ID = "select max(id) from eb_merchant";
    private static final String SQL_QUERY_PREFIX_COUNT = "select count(id) total from eb_merchant where 1=1";
    private static final String SQL_QUERY_PREFIX_LIST = "select * from eb_merchant where 1=1";
}
