package com.ishop.merchant.service;

import com.ishop.merchant.OrderConstants;
import com.ishop.model.po.EbMerchantOrder;
import com.ishop.model.po.EbOrder;
import com.ishop.model.po.EbOrderDetail;
import com.ishop.model.po.EbOrderInvoice;
import com.ishop.model.po.EbOrderInvoiceDetail;
import com.walker.db.page.GenericPager;
import com.walker.infrastructure.time.TimeRange;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.jdbc.service.BaseServiceImpl;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Service
public class OrderServiceImpl extends BaseServiceImpl {

    private static final String SQL_PAGE_USER_LIST = "select * from eb_order where uid=:uid";
    private static final String SQL_PAGE_PLATFORM_ORDER = "select id,mer_id,order_no,uid,pay_price,pay_type,paid,status" +
            ",refund_status,is_user_del,is_merchant_del,cancel_status,level,type,create_time from eb_order where 1=1";

    /**
     * 发货（express）数据库更新。
     * @param orderInvoice
     * @param orderInvoiceDetailList
     * @param orderDetailUpdateList
     * @param updateOrder
     * @date 2023-08-21
     */
    public void execSendExpress(EbOrderInvoice orderInvoice
            , List<EbOrderInvoiceDetail> orderInvoiceDetailList, List<EbOrderDetail> orderDetailUpdateList, EbOrder updateOrder){
        // 1:写入发货单
        // 2:写入发货单明细
        // 3:更新订单明细表中，邮递数量（delivery_num）字段
        // 4:更新商户订单快递方式为(express)，暂时无需设置，默认就是
        // 5:更新平台订单为：已发货
        this.insert(orderInvoice);
        this.insert(orderInvoiceDetailList);
        this.update(orderDetailUpdateList);
        this.update(updateOrder);
    }

    /**
     * 分页查询平台订单
     * @param merId 商户ID，可选
     * @param orderNo 订单编号，可选
     * @param type 订单类型，可选
     * @param timeRange 时间范围，可选
     * @param status 订单状态，可选
     * @return
     * @date 2023-07-20
     */
    public GenericPager<EbOrder> queryPagePlatformOrderList(Integer merId, String orderNo, Integer type, TimeRange timeRange, String status){
        Map<String, Object> param = new HashMap<>(4);
        StringBuilder sql = new StringBuilder(SQL_PAGE_PLATFORM_ORDER);
        if(merId != null && merId > 0){
            sql.append(" and mer_id=:merId");
            param.put("merId", merId);
        }
        if(StringUtils.isNotEmpty(orderNo)){
            sql.append(" and order_no=:orderNo");
            param.put("orderNo", orderNo);
        }
        if(type != null && type >= 0){
            sql.append(" and type=:type");
            param.put("type", type);
        }
        if(timeRange != null){
            if(timeRange.getStartTime() != null){
                sql.append(" and create_time >= :startTime");
                param.put("startTime", timeRange.getStartTime());
            }
            if(timeRange.getEndTime() != null){
                sql.append(" and create_time <= :endTime");
                param.put("endTime", timeRange.getEndTime());
            }
        }
        if(StringUtils.isNotEmpty(status)){
            this.setupStatusCondition(sql, status);
        }
        sql.append(" order by id desc");
        return this.selectSplit(sql.toString(), param, new EbOrder());
    }

    private void setupStatusCondition(StringBuilder sql, String status){
        //未支付
        if(status.equals(OrderConstants.MERCHANT_ORDER_STATUS_UNPAID)){
            sql.append(" and paid=0");
            sql.append(" and status=").append(OrderConstants.ORDER_STATUS_WAIT_PAY);
            sql.append(" and cancel_status=").append(OrderConstants.ORDER_CANCEL_STATUS_NORMAL);
            sql.append(" and is_user_del=0");

        } else if(status.equals(OrderConstants.MERCHANT_ORDER_STATUS_NOT_SHIPPED)){
            // 未发货
            sql.append(" and (status=").append(OrderConstants.ORDER_STATUS_WAIT_SHIPPING)
                    .append(" or status=").append(OrderConstants.ORDER_STATUS_PART_SHIPPING).append(")");
            sql.append(" and cancel_status=").append(OrderConstants.ORDER_CANCEL_STATUS_NORMAL);
            sql.append(" and refund_status<>").append(OrderConstants.ORDER_REFUND_STATUS_REFUND);
            sql.append(" and is_user_del=0");

        } else if(status.equals(OrderConstants.MERCHANT_ORDER_STATUS_SPIKE)){
            //待收货
            sql.append(" and status=").append(OrderConstants.ORDER_STATUS_WAIT_RECEIPT);
            sql.append(" and cancel_status=").append(OrderConstants.ORDER_CANCEL_STATUS_NORMAL);
            sql.append(" and refund_status<>").append(OrderConstants.ORDER_REFUND_STATUS_REFUND);
            sql.append(" and is_user_del=0");

        } else if(status.equals(OrderConstants.MERCHANT_ORDER_STATUS_RECEIVING)){
            //已收货
            sql.append(" and status=").append(OrderConstants.ORDER_STATUS_TAKE_DELIVERY);
            sql.append(" and cancel_status=").append(OrderConstants.ORDER_CANCEL_STATUS_NORMAL);
            sql.append(" and refund_status<>").append(OrderConstants.ORDER_REFUND_STATUS_REFUND);
            sql.append(" and is_user_del=0");

        } else if(status.equals(OrderConstants.MERCHANT_ORDER_STATUS_COMPLETE)){
            //交易完成
            sql.append(" and status=").append(OrderConstants.ORDER_STATUS_COMPLETE);
            sql.append(" and cancel_status=").append(OrderConstants.ORDER_CANCEL_STATUS_NORMAL);
            sql.append(" and refund_status<>").append(OrderConstants.ORDER_REFUND_STATUS_REFUND);
            sql.append(" and is_user_del=0");

        } else if(status.equals(OrderConstants.MERCHANT_ORDER_STATUS_AWAIT_VERIFICATION)){
            //待核销
            sql.append(" and status=").append(OrderConstants.ORDER_STATUS_AWAIT_VERIFICATION);
            sql.append(" and cancel_status=").append(OrderConstants.ORDER_CANCEL_STATUS_NORMAL);
            sql.append(" and refund_status<>").append(OrderConstants.ORDER_REFUND_STATUS_REFUND);
            sql.append(" and is_user_del=0");

        } else if(status.equals(OrderConstants.MERCHANT_ORDER_STATUS_REFUNDED)){
            //已退款
            sql.append(" and paid=1");
            sql.append(" and cancel_status=").append(OrderConstants.ORDER_CANCEL_STATUS_NORMAL);
            sql.append(" and refund_status=").append(OrderConstants.ORDER_REFUND_STATUS_REFUND);
            sql.append(" and is_user_del=0");

        } else if(status.equals(OrderConstants.MERCHANT_ORDER_STATUS_DELETED)){
            sql.append(" and is_user_del=1");
        }
    }

    /**
     * 根据主订单编号，返回一个商户订单信息。
     * <p>原：（支付完成进行商户拆单后可用）</p>
     * @param orderNo
     * @return
     * @date 2023-07-20
     */
    public EbMerchantOrder queryMerchantOrderOne(String orderNo){
        List<EbMerchantOrder> list = this.select("select * from eb_merchant_order where order_no=? limit 1", new Object[]{orderNo}, new EbMerchantOrder());
        if(StringUtils.isEmptyList(list)){
            return null;
        }
        return list.get(0);
    }

    /**
     * 根据平台订单编号，查询商户订单列表，商户订单（可能多商品）对应一个平台订单。
     * @param orderNo
     * @return
     * @date 2023-07-13
     */
    public List<EbMerchantOrder> queryMerchantOrderList(String orderNo){
        EbMerchantOrder param = new EbMerchantOrder();
        param.setOrderNo(orderNo);
        return this.select(param);
    }

    /**
     * 根据订单编号，返回订单信息。
     * @param orderNo
     * @return
     * @date 2023-07-12
     */
    public EbOrder queryOrder(String orderNo){
        EbOrder order = new EbOrder();
        order.setOrderNo(orderNo);
        List<EbOrder> list = this.select(order);
        if(StringUtils.isEmptyList(list)){
            return null;
        }
        return list.get(0);
    }

    /**
     * 平台订单，如果用订单编号（不用主键），还要配合商户ID才能确定一条记录。
     * @param orderNo
     * @param merId
     * @return
     * @date 2023-08-18
     */
    public EbOrder queryPlatformOrder(String orderNo, int merId){
        EbOrder order = new EbOrder();
        order.setOrderNo(orderNo);
        order.setMerId(merId);
        List<EbOrder> list = this.select(order);
        if(StringUtils.isEmptyList(list)){
            return null;
        }
        return list.get(0);
    }

    public List<EbOrderDetail> queryOrderDetailList(String orderNo){
        EbOrderDetail detail = new EbOrderDetail();
        detail.setOrderNo(orderNo);
        return this.select(detail);
    }

    /**
     * 根据平台订单ID，查询对应商品ID集合。
     * @param orderNo
     * @return
     * @date 2023-09-09
     */
    public List<Long> queryOrderProductIds(String orderNo){
        List<Map<String, Object>> list = this.select(SQL_ORDER_PRODUCT_ID, new Object[]{orderNo});
        if(StringUtils.isEmptyList(list)){
            return null;
        }
        List<Long> data = new ArrayList<>(2);
        for(Map<String, Object> map : list){
            data.add(Long.parseLong(map.get("product_id").toString()));
        }
        return data;
    }

    private static final String SQL_ORDER_PRODUCT_ID = "select product_id from eb_order_detail where order_no=?";

    /**
     * 订单发货获取订单详情列表
     * @param orderNo
     * @return
     * @date 2023-08-21
     */
    public List<EbOrderDetail> queryShipmentList(String orderNo){
        return this.select(SQL_SHIP_ORDER_DETAIL, new Object[]{orderNo}, new EbOrderDetail());
    }

    private static final String SQL_SHIP_ORDER_DETAIL = "select id,product_name,image,sku,pay_num,delivery_num,refund_num from eb_order_detail where order_no=?";

    /**
     * 分页返回用户订单列表
     * @param userId
     * @param status 订单状态（-1：全部，0：待支付，1：待发货,2：部分发货， 3：待核销，4：待收货,5：已收货,6：已完成，9：已取消）
     * @return
     * @date 2023-07-07
     */
    public GenericPager<EbOrder> queryUserOrderList(long userId, Integer status){
        StringBuilder sql = new StringBuilder(SQL_PAGE_USER_LIST);
        Map<String, Object> param = new HashMap<>(8);
        param.put("uid", userId);

        if(status != null && status.intValue() >= 0){
            if(status.intValue() == 1){
                sql.append(" and (status=").append(OrderConstants.ORDER_STATUS_WAIT_SHIPPING);
                sql.append(" or status=").append(OrderConstants.ORDER_STATUS_PART_SHIPPING).append(")");
            } else {
                sql.append(" and status=:status");
                param.put("status", status);
            }
        }
        // 退款状态
        sql.append(" and refund_status<").append(OrderConstants.ORDER_REFUND_STATUS_REFUND);
        sql.append(" and is_user_del=0");
        sql.append(" and is_merchant_del=0");
        sql.append(" and (type=").append(OrderConstants.ORDER_TYPE_NORMAL)
                .append(" or type=").append(OrderConstants.ORDER_TYPE_SECKILL).append(")");
        sql.append(" and is_del=0");
        sql.append(" order by id desc");
        return this.selectSplit(sql.toString(), param, new EbOrder());
    }
}
