package com.ishop.merchant.service;

import com.iplatform.base.VariableConstants;
import com.iplatform.base.service.UserServiceImpl;
import com.iplatform.model.po.S_user_core;
import com.ishop.merchant.BalanceRecordConstants;
import com.ishop.merchant.Constants;
import com.ishop.merchant.pojo.UserSearchParam;
import com.ishop.merchant.util.VoUtils;
import com.ishop.model.po.EbUser;
import com.ishop.model.po.EbUserBalanceRecord;
import com.ishop.model.po.EbUserConfig;
import com.ishop.model.po.EbUserToken;
import com.walker.db.page.GenericPager;
import com.walker.infrastructure.utils.DateUtils;
import com.walker.infrastructure.utils.MD5;
import com.walker.infrastructure.utils.NumberFormatUtils;
import com.walker.infrastructure.utils.NumberGenerator;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.jdbc.service.BaseServiceImpl;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 演示：用户注册管理数据库操作。
 * @author 时克英
 * @date 2023-06-16
 */
@Service
public class UserRegisterServiceImpl extends BaseServiceImpl {

    /**
     * 根据用户ID集合，返回对象集合。注意：不能数量过多。
     * @param userIds
     * @return
     * @date 2023-09-25
     */
    public List<EbUser> queryUserList(List<Long> userIds){
        if(userIds.size() > 64){
            throw new UnsupportedOperationException("批量查询用户过大，请业务调用调整参数。");
        }
        Map<String, Object> param = new HashMap<>(4);
        param.put("userIds", userIds);
        return this.select("select * from eb_user where id in (:userIds)", param, new EbUser());
    }

    /**
     * 批量设置用户标签。
     * @param userIds
     * @param tagIds
     * @date 2023-09-25
     */
    public void execBatchTag(List<Long> userIds, String tagIds){
//        Map<String, Object> param = new HashMap<>(4);
//        param.put("userIds", userIds);
//        this.update(new EbUser(), "where ", param);
        List<Object[]> param = new ArrayList<>(userIds.size());
        Object[] one;
        for(long userId : userIds){
            one = new Object[2];
            one[0] = tagIds;
            one[1] = userId;
            param.add(one);
        }
        this.execBatchUpdate(SQL_SET_TAG, param);
    }
    private static final String SQL_SET_TAG = "update eb_user set tag_id=? where id=?";

    /**
     * 更新（设置）用户，移动端用户商户管理员角色。
     * @param type 类型，见枚举<code>UserMerchantType</code>
     * @param roleId
     * @param merId
     * @param userId
     * @date 2023-09-25
     */
    public void execRoleMerchantAdmin(int type, long roleId, long merId, long userId){
        // 1:先删除移动端用户所在角色id
        // 2:添加移动端用户设置角色
        // 3:设置用户：type、merId
        this.execute("delete from s_role_user where user_id=?", new Object[]{userId});
        this.execute("insert into s_role_user(user_id, role_id) values(?,?)", new Object[]{userId, roleId});
        S_user_core userCore = new S_user_core(userId);
        userCore.setType(type);
        userCore.setMer_id(merId);
        this.update(userCore);
    }

    /**
     * 后台为用户变动余额
     * @param userId 用户ID
     * @param money 金额，元，保留2小数
     * @param type 充值 或 减扣
     * @param oldBalance 原始余额
     * @date 2023-09-06
     */
    public void execModifyUserBalance(long userId, double money, String type, double oldBalance){
        this.insertUserBalanceRecord(userId, money, type, oldBalance);
        this.execUpdateMoney(userId, money, type);
    }

    private void insertUserBalanceRecord(long userId, double payPrice, String type, double oldBalance){
        EbUserBalanceRecord userBalanceRecord = new EbUserBalanceRecord();
        userBalanceRecord.setId(NumberGenerator.getLongSequenceNumber());
        userBalanceRecord.setCreateTime(DateUtils.getDateTimeNumber());
        userBalanceRecord.setUid(userId);
        userBalanceRecord.setLinkId("0");
        userBalanceRecord.setLinkType(BalanceRecordConstants.BALANCE_RECORD_LINK_TYPE_SYSTEM);
        userBalanceRecord.setAmount(payPrice);
        if (type.equals(Constants.OPERATION_TYPE_ADD)) {
            // 增加
            userBalanceRecord.setType(BalanceRecordConstants.BALANCE_RECORD_TYPE_ADD);
            userBalanceRecord.setBalance(NumberFormatUtils.scaleAccuracy2(oldBalance + payPrice));
            userBalanceRecord.setRemark(BalanceRecordConstants.BALANCE_RECORD_REMARK_SYSTEM_ADD + payPrice);
        } else {
            // 扣减
            userBalanceRecord.setType(BalanceRecordConstants.BALANCE_RECORD_TYPE_SUB);
            userBalanceRecord.setBalance(NumberFormatUtils.scaleAccuracy2(oldBalance - payPrice));
            userBalanceRecord.setRemark(BalanceRecordConstants.BALANCE_RECORD_REMARK_SYSTEM_SUB + payPrice);
        }
        // 余额签名字段，金额保留2位小数，2023-08-30
        StringBuilder sb = new StringBuilder(String.valueOf(userId));
        sb.append(NumberFormatUtils.scaleAccuracy2(payPrice)).append(NumberFormatUtils.scaleAccuracy2(userBalanceRecord.getBalance()));
        userBalanceRecord.setSign(MD5.sign(sb.toString(), VariableConstants.TOKEN_SECRET, StringUtils.DEFAULT_CHARSET_UTF8));
        userBalanceRecord.setMonth(Integer.parseInt(DateUtils.getYearMonthCurrentValue()));
        this.insert(userBalanceRecord);
    }

    /**
     * 更新用户余额。
     * @param userId
     * @param money
     * @param type
     * @date 2023-07-12
     */
    public void execUpdateMoney(long userId, double money, String type){
        if (type.equals(Constants.OPERATION_TYPE_ADD)) {
            this.execute("update eb_user set now_money = now_money + ? where id=?", new Object[]{money, userId});
        } else if(type.equals(Constants.OPERATION_TYPE_SUBTRACT)){
            this.execute("update eb_user set now_money = now_money - ? where id=?", new Object[]{money, userId});
        } else {
            throw new UnsupportedOperationException("不支持的余额更新方式：" + type);
        }
    }

    /**
     * 更新用户积分
     *
     * @param uid      用户ID
     * @param integral 积分
     * @param type     增加add、扣减sub
     * @return int 成功更新记录条数
     * @date 2023-07-09
     */
    public int execUpdateIntegral(long uid, int integral, String type){
        Map<String, Object> param = new HashMap<>(2);
        StringBuilder sql = new StringBuilder("update eb_user");
        if (type.equals(Constants.OPERATION_TYPE_ADD)) {
            sql.append(" set integral=integral+:integral");
            param.put("integral", integral);
        } else {
            sql.append(" set integral=integral-:integral");
//            wrapper.apply(StrUtil.format(" integral - {} >= 0", integral));
            param.put("integral", integral);
        }
        sql.append(" where id=:id");
        param.put("id", uid);
        return this.execute(sql.toString(), param);
    }

    /**
     * 用户编辑个人信息。
     * @param ebUser
     * @param userCore
     * @date 2023-07-02
     */
    public void execUpdateUserEdit(EbUser ebUser, S_user_core userCore){
        this.save(ebUser);
        this.save(userCore);
    }

    /**
     * 微信等第三方登录是，已存在用户，但未绑定token，更新绑定信息。
     * @param userCore
     * @param spreadId 推广人ID
     * @param userTokenType token类型：
     * @param token openId等第三方参数
     * @date 2023-08-07
     */
    public void execUpdateBindUser(S_user_core userCore, Long spreadId, int userTokenType, String token){
        // 1-更新用户信息中：绑定信息
        // 2-更新注册用户信息绑定字段
        // 3-写入token绑定表数据
        // 4-绑定推广关系
        userCore.setUpdate_time(DateUtils.getDateTimeNumber());
        userCore.setBind_mobile(1);

        EbUser ebUser = new EbUser(userCore.getId());
        ebUser.setIsWechatIos(userCore.getIs_wechat_ios());
        ebUser.setIsWechatAndroid(userCore.getIs_wechat_android());
        ebUser.setIsWechatPublic(userCore.getIs_wechat_public());
        ebUser.setIsWechatRoutine(userCore.getIs_wechat_routine());
        ebUser.setUpdateTime(userCore.getUpdate_time());

        EbUserToken userToken = VoUtils.acquireEbUserToken(userTokenType, userCore.getId(), token);

        this.update(userCore);
        this.update(ebUser);
        this.insert(userToken);
        if(spreadId != null && spreadId > 0){
            // 绑定推广关系
//            bindSpread(finalUser, spreadPid);
        }
    }


    /**
     * 移动端手机（登录不存在手机号时）注册用户。
     * @param user_core
     * @param ebUser
     * @param roleId 角色ID，可选
     * @date 2023-06-30
     */
    public void execInsertMobileUserRegister(S_user_core user_core, EbUser ebUser, Long roleId, EbUserToken userToken){
//        long userId = NumberGenerator.getLongSequenceNumber();
//        user_core.setId(userId);
        long userId = user_core.getId();
        ebUser.setId(userId);
        this.insert(user_core);
        this.insert(ebUser);

        // 2023-08-06 增加用户配置表
        EbUserConfig config = new EbUserConfig(userId);
        this.insert(config);
        if(roleId != null && roleId.longValue() > 0){
            this.execute(UserServiceImpl.SQL_INSERT_ROLE_USER, new Object[]{userId, roleId});
        }

        // 2023-8-07
        if(userToken != null){
            this.insert(userToken);
        }
    }

    /**
     * 收藏过商户的注册用户列表，商户管理的用户。
     * @param merId
     * @param nickName
     * @param phone
     * @param registerType
     * @return
     * @date 2023-06-18
     */
    public GenericPager<EbUser> queryPageMerchantUserList(int merId, String nickName, String phone, String registerType){
        Map<String, Object> parameter = new HashMap<>(4);
        StringBuilder sql = new StringBuilder(SQL_PAGE_MER_USER);
        parameter.put("merId", merId);
        if(StringUtils.isNotEmpty(phone)){
            sql.append(" and u.phone = :phone");
            parameter.put("phone", phone);
        }
        if(StringUtils.isNotEmpty(registerType)){
            sql.append(" and u.register_type = :registerType");
            parameter.put("registerType", registerType);
        }
        if(StringUtils.isNotEmpty(nickName)){
            sql.append(" and u.nickname = :nickname");
            parameter.put("nickname", StringUtils.CHAR_PERCENT + nickName + StringUtils.CHAR_PERCENT);
        }
        return this.selectSplit(sql.toString(), parameter, new EbUser());
    }

    public GenericPager<EbUser> queryPageUserList(UserSearchParam param){
        Map<String, Object> parameter = new HashMap<>(4);
        StringBuilder sql = new StringBuilder(SQL_PAGE_USER_PREFIX);
        if(param != null){
            if(StringUtils.isNotEmpty(param.getPhone())){
                sql.append(" and phone=:phone");
                parameter.put("phone", param.getPhone());
            } else if(StringUtils.isNotEmpty(param.getNikename())){
                sql.append(" and nickname like :nickName");
                parameter.put("nickName", StringUtils.CHAR_PERCENT + param.getNikename() + StringUtils.CHAR_PERCENT);
            }
        }
        return this.selectSplit(sql.toString(), parameter, new EbUser());
    }

    @Deprecated
    public GenericPager<EbUser> queryPageUser02(EbUser user){
//        EbUser user = new EbUser();
        if(StringUtils.isNotEmpty(user.getPhone())){
            user.setPhone(user.getPhone());
        }
        if(StringUtils.isNotEmpty(user.getNickname())){
            user.setNickname(StringUtils.CHAR_PERCENT + user.getNickname() + StringUtils.CHAR_PERCENT);
        }
        return this.selectSplit(user);
    }

    private static final String SQL_PAGE_MER_USER = "SELECT u.* FROM eb_user u RIGHT JOIN eb_user_merchant_collect um on u.id = um.uid where um.mer_id=:merId";
    private static final String SQL_PAGE_USER_PREFIX = "select * from eb_user where 1=1";
}
