package com.ishop.merchant.support;

import com.iplatform.base.VariableConstants;
import com.iplatform.core.BeanContextAware;
import com.iplatform.model.po.S_pay_notify;
import com.iplatform.model.po.S_pay_order;
import com.iplatform.pay.support.DefaultWechatV2OrderCallback;
import com.ishop.merchant.BalanceRecordConstants;
import com.ishop.merchant.OrderConstants;
import com.ishop.merchant.PayConstants;
import com.ishop.merchant.UserRegCache;
import com.ishop.merchant.service.PayServiceImpl;
import com.ishop.merchant.util.PayUtils;
import com.ishop.model.po.EbOrder;
import com.ishop.model.po.EbRechargeOrder;
import com.ishop.model.po.EbUser;
import com.ishop.model.po.EbUserBalanceRecord;
import com.walker.infrastructure.utils.DateUtils;
import com.walker.infrastructure.utils.NumberGenerator;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.pay.NotifyValue;
import com.walker.pay.PayStatus;
import com.walker.pay.wechat.Constants;
import com.walker.pay.wechat.v2.NotifyOrder;

public class MerchantWechatOrderCallback extends DefaultWechatV2OrderCallback {

    /**
     * 订单通知：更新系统订单状态，写入平台通知记录。
     * <pre>
     *     重写支付平台中，通知订单完成方法：
     *     1) 更新电商模块，订单支付状态
     *     2) 对于虚拟商品，自动发货，更新订单为'已完成'
     * </pre>
     * @param notifyValue
     * @date 2023-07-08
     * @date 2023-09-09 虚拟商品，自动发货，更新订单为'已完成'
     */
    @Override
    protected void doNotifyOrderPay(NotifyValue<NotifyOrder> notifyValue){
        long orderId = Long.parseLong(notifyValue.getOrderId());
        S_pay_notify exist = this.getPayService().get(new S_pay_notify(orderId));
        if(exist != null && exist.getPay_status().equals(PayStatus.Success.getIndex())){
            logger.warn("订单通知已经接收，而且支付成功，不再重复处理，notifyId = {}, orderId = {}", notifyValue.getId(), orderId);
            return;
        }
        S_pay_notify s_pay_notify = this.acquirePayNotify(notifyValue);

        // 2023-09-12 根据平台订单判断是哪种支付：消费、充值、退款等
        S_pay_order platformOrder = this.getPayService().get(new S_pay_order(orderId));
        String attach = platformOrder.getAttach();
//        String attach = notifyValue.getData().getAttach();
        String[] attachArray = StringUtils.commaDelimitedListToStringArray(attach);
        String payServiceType = attachArray[0];
//        String userId = attachArray[1];
        // 2023-08-28 由于微信通知没有返回attach，所以这里直接查支付订单，如果退款则不再这里处理！

        if(payServiceType.equals(PayConstants.PAY_SERVICE_TYPE_ORDER)){
            this.processOrder(orderId, notifyValue, s_pay_notify, exist);

        } else if(payServiceType.equals(PayConstants.PAY_SERVICE_TYPE_RECHARGE)){
            this.processRecharge(orderId, s_pay_notify, exist);

        } else {
            throw new UnsupportedOperationException("未实现支付通知回调，payServiceType=" + payServiceType);
        }

//        if(payServiceType.equals(PayConstants.PAY_SERVICE_TYPE_ORDER)){
//            // 订单通知
//            EbOrder ebOrder = this.getPayService().get(new EbOrder(orderId));
//            ebOrder.setOutTradeNo(notifyValue.getTradeNo());    // 微信订单id
//            this.getPayService().execPayWechatPublic(ebOrder, s_pay_notify, exist != null);
//            logger.debug("微信通知更新订单，orderId={}", orderId);
//        } else {
//            // 其他通知
//            logger.warn("微信v2，其他通知，未实现代码!");
//        }
    }

    private void processRecharge(long orderId, S_pay_notify s_pay_notify, S_pay_notify exist){
        EbRechargeOrder rechargeOrder = this.getPayService().get(new EbRechargeOrder(orderId));
        rechargeOrder.setPaid(1);
        rechargeOrder.setPayTime(DateUtils.getDateTimeNumber());
        rechargeOrder.setUpdateTime(rechargeOrder.getPayTime());

        long userId = rechargeOrder.getUid();
        double addPrice = rechargeOrder.getPrice() + rechargeOrder.getGivePrice();

        EbUser user = BeanContextAware.getBeanByType(UserRegCache.class).get(userId);
        double balance = user.getNowMoney() + addPrice;

        // 余额变动对象
        EbUserBalanceRecord record = new EbUserBalanceRecord();
        record.setId(NumberGenerator.getLongSequenceNumber());
        record.setCreateTime(DateUtils.getDateTimeNumber());
        record.setUid(userId);
//        record.setLinkId(rechargeOrder.getOrderNo());
        record.setLinkId(String.valueOf(orderId));
        record.setLinkType(BalanceRecordConstants.BALANCE_RECORD_LINK_TYPE_RECHARGE);
        record.setType(BalanceRecordConstants.BALANCE_RECORD_TYPE_ADD);
        record.setAmount(addPrice);
        record.setBalance(balance);
        record.setRemark(BalanceRecordConstants.BALANCE_RECORD_REMARK_RECHARGE + addPrice);
        record.setSign(PayUtils.acquireUserBalanceSign(userId, addPrice, balance, VariableConstants.TOKEN_SECRET));
        record.setMonth(Integer.parseInt(DateUtils.getYearMonthCurrentValue()));

        this.getPayService().execRechargeWechatNotify(rechargeOrder, record, addPrice, s_pay_notify, exist != null);
        // 更新缓存中用户余额，为外面更新准备
        user.setNowMoney(user.getNowMoney() + addPrice);
        logger.debug("微信通知更新（充值）订单，orderId={}", orderId);
    }

    private void processOrder(long orderId, NotifyValue<NotifyOrder> notifyValue, S_pay_notify s_pay_notify, S_pay_notify exist){
        EbOrder ebOrder = this.getPayService().get(new EbOrder(orderId));
        ebOrder.setOutTradeNo(notifyValue.getTradeNo());    // 微信订单id
        ebOrder.setPayType(PayConstants.PAY_TYPE_WE_CHAT);
        ebOrder.setUpdateTime(DateUtils.getDateTimeNumber());

        String tradeType = notifyValue.getTradeType();
        if(StringUtils.isEmpty(tradeType)){
            ebOrder.setPayChannel("none");
        } else if(tradeType.equals(Constants.PAY_TYPE_H5)){
            ebOrder.setPayChannel(PayConstants.PAY_CHANNEL_WECHAT_PUBLIC);
        } else if(tradeType.equals(Constants.PAY_TYPE_APP)){
            ebOrder.setPayChannel(PayConstants.PAY_CHANNEL_WECHAT_APP_ANDROID);
        } else if(tradeType.equals(Constants.PAY_TYPE_NATIVE)){
            ebOrder.setPayChannel(PayConstants.PAY_CHANNEL_WECHAT_MINI);
        } else {
            ebOrder.setPayChannel(tradeType);
        }

        // 2023-09-09
        if(PayUtils.isAutoShippingDone(ebOrder.getOrderNo())){
            // 当余额支付是，虚拟商品自动完成订单，无需发货
            ebOrder.setStatus(OrderConstants.ORDER_STATUS_COMPLETE);
        }

        this.getPayService().execPayWechatPublic(ebOrder, s_pay_notify, exist != null);
        logger.debug("微信通知更新订单，orderId={}", orderId);
    }

    private PayServiceImpl getPayService(){
        return BeanContextAware.getBeanByType(PayServiceImpl.class);
    }
}
