package com.ishop.merchant.util;

import com.iplatform.core.BeanContextAware;
import com.ishop.merchant.Constants;
import com.ishop.merchant.ProductCache;
import com.ishop.merchant.service.OrderServiceImpl;
import com.walker.infrastructure.utils.DateUtils;
import com.walker.infrastructure.utils.MD5;
import com.walker.infrastructure.utils.NumberFormatUtils;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.pay.Order;
import com.walker.pay.PayType;
import com.walker.pay.ServiceProvider;
import com.walker.pay.support.DefaultOrder;

import java.util.List;

public class PayUtils {

    /**
     * 过滤微信openId中双引号。
     * <pre>
     *     不知道为啥openId字符串会带双引号，目前没找到原因，先直接剔除，后面要找到（初步估计是redis取出来带的）
     * </pre>
     * @param userOpenId
     * @return
     * @date 2023-08-28
     */
    public static final String filterUserOpenId(String userOpenId){
        if(StringUtils.isNotEmpty(userOpenId) && userOpenId.indexOf("\"") >= 0){
            return userOpenId.replaceAll("\"", StringUtils.EMPTY_STRING);
        } else {
            return userOpenId;
        }
    }

    /**
     * 余额签名字段，金额保留2位小数
     * @param userId
     * @param payPrice
     * @param balance
     * @param md5Key
     * @return
     * @date 2023-08-30
     */
    public static final String acquireUserBalanceSign(long userId, double payPrice, double balance, String md5Key){
        StringBuilder sb = new StringBuilder(String.valueOf(userId));
        sb.append(NumberFormatUtils.scaleAccuracy2(payPrice)).append(NumberFormatUtils.scaleAccuracy2(balance));
        return MD5.sign(sb.toString(), md5Key, StringUtils.DEFAULT_CHARSET_UTF8);
    }

    /**
     * 生成第三方支付订单编号。
     * @param payType
     * @return
     * @date 2023-09-12
     */
    public static final String acquirePayOutTradeNo(String payType){
        return new StringBuilder(payType)
                .append(System.currentTimeMillis()).append(StringUtils.generateRandomNumber(6)).toString();
    }

    /**
     * 检测订单中商品是否（全部）为虚拟商品，如果是则无需发货，订单直接完成。
     * @param orderNo
     * @return
     * @date 2023-09-09
     */
    public static final boolean isAutoShippingDone(String orderNo){
        List<Long> productIds = BeanContextAware.getBeanByType(OrderServiceImpl.class).queryOrderProductIds(orderNo);
        if(StringUtils.isEmptyList(productIds)){
            throw new IllegalStateException("未查询到订单明细商品，orderNo=" + orderNo);
        }
        int templateId = 0;
        for(long productId : productIds){
            templateId = BeanContextAware.getBeanByType(ProductCache.class).get(productId).getTempId();
            if(templateId != Constants.SHIPPING_TEMPLATE_ID_NO_SEND){
                return false;
            }
        }
        return true;
    }

    /**
     * 返回微信支付v2版本需要的平台订单记录
     * @param moneyFen
     * @param orderId
     * @param title
     * @param attach
     * @param ip
     * @param notifyUrl
     * @param userOpenId 用户对应商户appid下的微信openId，2023-08-28
     * @return
     * @date 2023-08-10
     */
    public static final Order acquirePlatformOrderH5WechatV2(long moneyFen, long orderId
            , String title, String attach, String ip, String notifyUrl, String userOpenId){
        DefaultOrder order = new DefaultOrder();
        order.setServiceProvider(ServiceProvider.Wechat);
        order.setVersion("v2");
        order.setId(orderId);
        order.setCreateTime(DateUtils.getDateTimeNumber(System.currentTimeMillis()));
        order.setPayType(PayType.H5);
        order.setPayDefinitionId("Wechat_v2");
        order.setTotalMoney(moneyFen);
//        order.setNotifyUrl("http://47.99.216.98:7003/pay/wx/notify.do");
        order.setNotifyUrl(notifyUrl);
        order.setIp(ip);
        order.setNonce(StringUtils.generateRandomNumber(6));
        order.setTitle(title);
        order.setAttach(attach);
        order.setBuyerId(userOpenId);
        return order;
    }

    /**
     * 获取微信小程序支付类型订单。
     * @param moneyFen
     * @param orderId
     * @param title
     * @param attach
     * @param ip
     * @param notifyUrl
     * @param userMiniOpenId
     * @return
     * @date 2023-09-15
     */
    public static final Order acquirePlatformOrderRoutineWechatV2(long moneyFen, long orderId
            , String title, String attach, String ip, String notifyUrl, String userMiniOpenId){
        DefaultOrder order = new DefaultOrder();
        order.setServiceProvider(ServiceProvider.Wechat);
        order.setVersion("v2");
        order.setId(orderId);
        order.setCreateTime(DateUtils.getDateTimeNumber(System.currentTimeMillis()));
        order.setPayType(PayType.Applet);   // 小程序支付类型订单
        order.setPayDefinitionId("Wechat_v2");
        order.setTotalMoney(moneyFen);
        order.setNotifyUrl(notifyUrl);
        order.setIp(ip);
        order.setNonce(StringUtils.generateRandomNumber(6));
        order.setTitle(title);
        order.setAttach(attach);
        order.setBuyerId(userMiniOpenId);
        return order;
    }
}
