package com.ishop.merchant.util;

import com.ishop.merchant.Constants;
import com.ishop.model.po.EbProduct;
import com.ishop.model.po.EbProductAttr;
import com.ishop.model.po.EbProductAttrValue;
import com.ishop.model.vo.ProductAddVo;
import com.walker.infrastructure.utils.JsonUtils;
import com.walker.infrastructure.utils.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

public class ProductAttrUtils {

    protected static final Logger logger = LoggerFactory.getLogger(ProductAttrUtils.class);

    /**
     * 根据价格规格，返回自定义属性中要存放的价格信息。
     * <pre>
     *
     * </pre>
     * @param priceAttr 几个规格定义，如：[100,200,300,500,其他金额]
     * @param attrValueSku 自定义属性值，如：98#,其他金额 或者 98#,500
     * @return
     * @date 2023-09-10
     * @date 2023-09-11 目前该方法未使用，因为通过规格分组已经可以配置出来金额了。
     */
    public static final double acquirePriceAttrMoney(String[] priceAttr, String attrValueSku){
        String[] skuArray = StringUtils.commaDelimitedListToStringArray(attrValueSku);
        for(String skuValue : skuArray){
            if(skuValue.equals(Constants.ATTR_VALUE_OTHER_PRICE) || skuValue.equals(Constants.ATTR_VALUE_CUSTOM_PRICE)){
                return 0;
            }
            for(String price : priceAttr){
                if(skuValue.equals(price)){
                    return Double.parseDouble(price);
                }
            }
        }
        return -1;
    }

    /**
     * 返回价格规格，对应的价格数组，如果存在的话。
     * <pre>
     *    格式如下：
     *    [100, 200, 其他金额（或自定义）]
     * </pre>
     * @param attrList
     * @return
     * @date 2023-09-10
     */
    public static final String[] hasPriceAttr(List<EbProductAttr> attrList){
        for(EbProductAttr attr : attrList){
            if(attr.getAttrName().equals(Constants.ATTR_NAME_PRICE)){
                return StringUtils.commaDelimitedListToStringArray(attr.getAttrValues());
            }
        }
        return null;
    }

    /**
     * 设置添加（或编辑）商品价格信息。
     * @param request
     * @param product
     * @date 2023-08-02
     */
    public static final void combineProductAttrValue(ProductAddVo request, EbProduct product){
        //计算价格
        List<EbProductAttrValue> attrValueAddRequestList = request.getAttrValue();
        EbProductAttrValue minAttrValue = attrValueAddRequestList.stream().min(Comparator.comparing(EbProductAttrValue::getPrice)).get();
        product.setPrice(minAttrValue.getPrice());
        product.setOtPrice(minAttrValue.getOtPrice());
        product.setCost(minAttrValue.getCost());
        product.setStock(attrValueAddRequestList.stream().mapToInt(EbProductAttrValue::getStock).sum());
    }

    /**
     * 商品sku。
     * <p>返回数据格式，如：92汽油,95汽油,98汽油，规格值拼接了一下，要存储数据库</p>
     *
     * @param attrValue json字符串，如：{"92号":"92号", "93号":"93号"}
     * @return sku
     */
    public static final String getSku(String attrValue) {
//        LinkedHashMap<String, String> linkedHashMap = JSONObject.parseObject(attrValue, LinkedHashMap.class, Feature.OrderedField);
//        LinkedHashMap<String, String> linkedHashMap = null;
//        logger.debug("======= " + attrValue);
        if(attrValue.startsWith("\"")){
            attrValue = attrValue.substring(1, attrValue.length()-1);
            // 前端传来的包含双引号转移反斜杠，老报错
            attrValue = attrValue.replace("\\", StringUtils.EMPTY_STRING);
        }
        Map<String, String> map = null;
        try {
            map = JsonUtils.jsonStringToObject(attrValue, Map.class);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        Iterator<Map.Entry<String, String>> iterator = map.entrySet().iterator();
        List<String> strings = new ArrayList<>(map.size());
        while (iterator.hasNext()) {
            Map.Entry<String, String> next = iterator.next();
            strings.add(next.getValue());
        }
        return StringUtils.collectionToCommaDelimitedString(strings);
    }

    /**
     * 把给定的 Key，分隔为：type | productId
     * @param key
     * @return
     * @date 2023-06-14
     */
    public static final String[] splitTypeAndProductId(String key){
        return StringUtils.commaDelimitedListToStringArray(key);
    }

    public static final String combineKey(int type, long productId){
        return new StringBuilder().append(type)
                .append(StringUtils.DEFAULT_SPLIT_SEPARATOR).append(productId).toString();
    }
}
