package com.ishop.mobile.util;

import com.iplatform.base.PlatformRuntimeException;
import com.iplatform.core.BeanContextAware;
import com.ishop.merchant.Constants;
import com.ishop.merchant.MerchantCache;
import com.ishop.merchant.OrderConstants;
import com.ishop.merchant.ProductConstants;
import com.ishop.merchant.service.ProductAttrValueServiceImpl;
import com.ishop.merchant.service.ProductServiceImpl;
import com.ishop.mobile.pojo.PreOrderDetailRequest;
import com.ishop.mobile.pojo.PreOrderRequest;
import com.ishop.mobile.service.CartServiceImpl;
import com.ishop.model.po.EbCart;
import com.ishop.model.po.EbMerchant;
import com.ishop.model.po.EbMerchantOrder;
import com.ishop.model.po.EbOrder;
import com.ishop.model.po.EbOrderDetail;
import com.ishop.model.po.EbProduct;
import com.ishop.model.po.EbProductAttrValue;
import com.ishop.model.po.EbUserAddress;
import com.ishop.model.request.OrderMerchantRequest;
import com.ishop.model.vo.PreMerchantOrderVo;
import com.ishop.model.vo.PreOrderDetailVo;
import com.ishop.model.vo.PreOrderVo;
import com.walker.infrastructure.arguments.ArgumentsManager;
import com.walker.infrastructure.utils.DateUtils;
import com.walker.infrastructure.utils.NumberFormatUtils;
import com.walker.infrastructure.utils.NumberGenerator;
import com.walker.infrastructure.utils.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

public class OrderUtils {

    private static final Logger logger = LoggerFactory.getLogger(OrderUtils.class);

    /**
     * 生成系统订单号。
     * @param prefix 前缀，如：pt/sh 等
     * @return
     * @date 2023-07-09
     */
    public static final String generateOrderNo(String prefix){
        return new StringBuilder(prefix).append(NumberGenerator.getLongSequenceNumber())
                .append(StringUtils.generateRandomNumber(4)).toString();
    }

    public static final Object[] acquireMerchantOrderList(List<Long> couponIdList
            , List<PreMerchantOrderVo> merchantOrderVoList, List<OrderMerchantRequest> orderMerchantRequestList
            , String orderNo, int orderType, long userId, long createTime, EbUserAddress userAddress, boolean autoShippingDone){
        Object[] result = new Object[2];
        List<EbMerchantOrder> merchantOrderList = new ArrayList<>(4);
        List<EbOrderDetail> orderDetailList = new ArrayList<>(4);
        for (PreMerchantOrderVo merchantOrderVo : merchantOrderVoList) {
            EbMerchantOrder merchantOrder = new EbMerchantOrder();
            merchantOrder.setId(NumberGenerator.getLongSequenceNumber());
            merchantOrder.setCreateTime(createTime);
            merchantOrder.setOrderNo(orderNo);
            merchantOrder.setMerId(merchantOrderVo.getMerId());
            merchantOrder.setUid(userId);
            for (OrderMerchantRequest om : orderMerchantRequestList) {
                if (om.getMerId().equals(merchantOrderVo.getMerId())) {
                    if (StringUtils.isNotEmpty(om.getRemark())) {
                        merchantOrder.setUserRemark(om.getRemark());
                    }
                    merchantOrder.setShippingType(om.getShippingType());
                    break;
                }
            }
            if (merchantOrder.getShippingType().intValue() == OrderConstants.ORDER_SHIPPING_TYPE_PICK_UP) {
                merchantOrder.setUserAddress(merchantOrderVo.getMerName());
                merchantOrder.setVerifyCode(StringUtils.generateRandomNumber(6));
            } else if(!autoShippingDone) {
                // 只有实物商品需要发货，虚拟商品不用
                merchantOrder.setRealName(userAddress.getRealName());
                merchantOrder.setUserPhone(userAddress.getPhone());
                String userAddressStr = userAddress.getProvince() + userAddress.getCity() + userAddress.getDistrict() + userAddress.getStreet() + userAddress.getDetail();
                merchantOrder.setUserAddress(userAddressStr);
            } else {
                merchantOrder.setUserAddress(Constants.SHIPPING_TEMPLATE_NAME_NO_SEND);
            }

            merchantOrder.setTotalNum(merchantOrderVo.getProTotalNum());
            merchantOrder.setProTotalPrice(merchantOrderVo.getProTotalFee());
            merchantOrder.setTotalPostage(merchantOrderVo.getFreightFee());
            merchantOrder.setTotalPrice(merchantOrder.getProTotalPrice() + merchantOrder.getTotalPostage());
            merchantOrder.setPayPostage(merchantOrder.getTotalPostage());
            merchantOrder.setUseIntegral(merchantOrderVo.getUseIntegral());
            merchantOrder.setIntegralPrice(merchantOrderVo.getIntegralPrice());
            merchantOrder.setCouponId(merchantOrderVo.getUserCouponId());
            if (merchantOrder.getCouponId() > 0) {
                couponIdList.add(merchantOrder.getCouponId());
            }
            merchantOrder.setCouponPrice(merchantOrderVo.getCouponFee());
            merchantOrder.setPayPrice(merchantOrder.getTotalPrice() - merchantOrder.getCouponPrice() - merchantOrder.getIntegralPrice());
            merchantOrder.setGainIntegral(0);
            merchantOrder.setType(orderType);
            merchantOrderList.add(merchantOrder);

            List<PreOrderDetailVo> detailVoList = merchantOrderVo.getOrderInfoList();
            for (PreOrderDetailVo detailVo : detailVoList) {
                EbOrderDetail orderDetail = new EbOrderDetail();
                orderDetail.setId(NumberGenerator.getLongSequenceNumber());
                orderDetail.setCreateTime(createTime);
                orderDetail.setOrderNo(orderNo);
                orderDetail.setMerId(merchantOrder.getMerId());
                orderDetail.setUid(userId);
                orderDetail.setProductId(detailVo.getProductId());
                orderDetail.setProductName(detailVo.getProductName());
                orderDetail.setImage(detailVo.getImage());
                orderDetail.setAttrValueId(detailVo.getAttrValueId());
                orderDetail.setSku(detailVo.getSku());
                orderDetail.setPrice(detailVo.getPrice());
                orderDetail.setPayNum(detailVo.getPayNum());
                orderDetail.setWeight(detailVo.getWeight());
                orderDetail.setVolume(detailVo.getVolume());
                orderDetail.setProductType(detailVo.getProductType());
                orderDetail.setSubBrokerageType(detailVo.getSubBrokerageType());
                orderDetail.setBrokerage(detailVo.getBrokerage());
                orderDetail.setBrokerageTwo(detailVo.getBrokerageTwo());
                orderDetail.setFreightFee(detailVo.getFreightFee());
                orderDetail.setCouponPrice(detailVo.getCouponPrice());
                orderDetail.setUseIntegral(detailVo.getUseIntegral());
                orderDetail.setIntegralPrice(detailVo.getIntegralPrice());
                orderDetail.setPayPrice(0d);
                double detailPayPrice = (orderDetail.getPrice().doubleValue() * orderDetail.getPayNum()) + orderDetail.getFreightFee() - orderDetail.getCouponPrice() - orderDetail.getIntegralPrice();
//                double detailPayPrice = orderDetail.getPrice().multiply(new BigDecimal(orderDetail.getPayNum().toString())).add(orderDetail.getFreightFee()).subtract(orderDetail.getCouponPrice()).subtract(orderDetail.getIntegralPrice());
                if (detailPayPrice >= 0) {
                    orderDetail.setPayPrice(detailPayPrice);
                }
                orderDetailList.add(orderDetail);
            }
        }
        result[0] = merchantOrderList;
        result[1] = orderDetailList;
        return result;
    }

    /**
     * 生成一个平台订单。
     * @param userId
     * @param preOrderVo
     * @return
     * @date 2023-07-09
     */
    public static final EbOrder acquirePlatformOrder(long userId, PreOrderVo preOrderVo){
        List<PreMerchantOrderVo> list = preOrderVo.getMerchantOrderVoList();
        EbOrder order = new EbOrder();
        order.setId(NumberGenerator.getLongSequenceNumber());
        order.setCreateTime(DateUtils.getDateTimeNumber());
        String orderNo = generateOrderNo(OrderConstants.ORDER_PREFIX_PLATFORM);
        order.setOrderNo(orderNo);
        order.setMerId(0);
        order.setUid(userId);
        order.setTotalNum(preOrderVo.getOrderProNum());
        order.setProTotalPrice(preOrderVo.getProTotalFee());
        order.setTotalPostage(preOrderVo.getFreightFee());
        order.setTotalPrice(order.getProTotalPrice() + order.getTotalPostage());
        order.setCouponPrice(preOrderVo.getCouponFee());
//        order.setUseIntegral(merchantOrderVoList.stream().mapToInt(PreMerchantOrderVo::getUseIntegral).sum());
        order.setUseIntegral(calculateTotalIntegral(list));
//        order.setIntegralPrice(list.stream().map(PreMerchantOrderVo::getIntegralPrice).reduce(BigDecimal.ZERO, BigDecimal::add));
        order.setIntegralPrice(calculateTotalIntegralPrice(list));
        order.setPayPrice(order.getProTotalPrice() + order.getTotalPostage() - order.getCouponPrice() - order.getIntegralPrice());
        order.setPayPostage(order.getTotalPostage());
        order.setPaid(0);
        order.setCancelStatus(OrderConstants.ORDER_CANCEL_STATUS_NORMAL);
        order.setLevel(OrderConstants.ORDER_LEVEL_PLATFORM);
        order.setType(preOrderVo.getType());// 默认普通订单
        return order;
    }

    private static final double calculateTotalIntegralPrice(List<PreMerchantOrderVo> merchantOrderVoList){
        double sum = 0;
        for(PreMerchantOrderVo vo: merchantOrderVoList){
            sum += vo.getIntegralPrice();
        }
        return sum;
    }

    private static final int calculateTotalIntegral(List<PreMerchantOrderVo> merchantOrderVoList){
        int sum = 0;
        for(PreMerchantOrderVo vo: merchantOrderVoList){
            sum += vo.getUseIntegral();
        }
        return sum;
    }

    /**
     * 校验预下单信息。
     * @param request
     * @param cdnUrl
     * @return
     * @date 2023-07-06
     */
    public static final PreOrderVo validatePreOrderRequest(PreOrderRequest request, String cdnUrl){
        PreOrderVo vo = new PreOrderVo();
        List<PreMerchantOrderVo> merchantOrderVoList = new ArrayList<>(4);
        String preOrderType = request.getPreOrderType();
        if(preOrderType.equals(Constants.PRE_ORDER_TYPE_SHOPPING_CART)){
            logger.info("预下单检查调用:购物车购买");
            merchantOrderVoList = validatePreOrderShopping(request, cdnUrl);
            List<Long> cartIdList = request.getOrderDetails().stream().map(PreOrderDetailRequest::getShoppingCartId).distinct().collect(Collectors.toList());
            vo.setCartIdList(cartIdList);

        } else if (preOrderType.equals(Constants.PRE_ORDER_TYPE_BUY_NOW)) {
            logger.info("预下单检查调用:立即购买");
            // 立即购买只会有一条详情
            PreOrderDetailRequest detailRequest = request.getOrderDetails().get(0);
            merchantOrderVoList.add(validatePreOrderNormal(detailRequest, cdnUrl));

        } else if (preOrderType.equals(Constants.PRE_ORDER_TYPE_VIDEO)) {
            logger.info("预下单检查调用:视频号下单");
            // 视频号暂时只能购买一个商品
            PreOrderDetailRequest detailRequest = request.getOrderDetails().get(0);
            merchantOrderVoList.add(validatePreOrderVideo(detailRequest));
            vo.setType(OrderConstants.ORDER_TYPE_VIDEO);

        } else if (preOrderType.equals(Constants.PRE_ORDER_TYPE_SECOND_KILL)) {
            logger.info("预下单检查调用:秒杀下单");
            // 秒杀只支持单商品支付
            PreOrderDetailRequest detailRequest = request.getOrderDetails().get(0);
//            merchantOrderVoList.add(seckillService.validatePreOrderSeckill(detailRequest));
            vo.setType(OrderConstants.ORDER_TYPE_SECKILL);

        } else {
            throw new UnsupportedOperationException("不支持的预下单类型：" + preOrderType);
        }
        vo.setMerchantOrderVoList(merchantOrderVoList);
        return vo;
    }

    private static final List<PreMerchantOrderVo> validatePreOrderShopping(PreOrderRequest request, String cdnUrl){
        List<PreMerchantOrderVo> merchantOrderVoList = new ArrayList<>(2);
        for(PreOrderDetailRequest detailRequest : request.getOrderDetails()){
            if(detailRequest.getShoppingCartId() == null){
                throw new PlatformRuntimeException("购物车编号不存在");
            }
            EbCart cart = BeanContextAware.getBeanByType(CartServiceImpl.class).get(new EbCart(detailRequest.getShoppingCartId()));
            if(cart == null){
                throw new PlatformRuntimeException("购物车信息不存在：" + detailRequest.getShoppingCartId());
            }
            detailRequest.setProductId(cart.getProductId());
            detailRequest.setAttrValueId(cart.getProductAttrUnique());
            detailRequest.setProductNum(cart.getCartNum());
            PreMerchantOrderVo merchantOrderVo = validatePreOrderNormal(detailRequest, cdnUrl);
            if (merchantOrderVoList.stream().anyMatch(m -> m.getMerId().equals(merchantOrderVo.getMerId()))) {
                for (PreMerchantOrderVo orderVo : merchantOrderVoList) {
                    if (orderVo.getMerId().equals(merchantOrderVo.getMerId())) {
                        orderVo.getOrderInfoList().addAll(merchantOrderVo.getOrderInfoList());
                        break;
                    }
                }
            } else {
                merchantOrderVoList.add(merchantOrderVo);
            }
        }
        return merchantOrderVoList;
    }

    /**
     * 普通商品下单校验
     *
     * @param detailRequest 商品参数
     */
    private static PreMerchantOrderVo validatePreOrderNormal(PreOrderDetailRequest detailRequest, String cdnUrl) {
        // 普通商品
        if (detailRequest.getProductId() == null) {
            throw new PlatformRuntimeException("商品编号不能为空");
        }
        if (detailRequest.getAttrValueId() == null) {
            throw new PlatformRuntimeException("商品规格属性值不能为空");
        }
        if (detailRequest.getProductNum() == null || detailRequest.getProductNum() <= 0) {
            throw new PlatformRuntimeException("购买数量必须大于0");
        }
        // 查询商品信息
        EbProduct product = BeanContextAware.getBeanByType(ProductServiceImpl.class).get(new EbProduct(detailRequest.getProductId()));
        if(product == null || product.getIsDel().intValue() == 1){
            throw new PlatformRuntimeException("商品信息不存在，请刷新后重新选择");
        }
        if(product.getIsShow().intValue() == 0){
            throw new PlatformRuntimeException("商品已下架，请刷新后重新选择");
        }
        if (product.getStock() < detailRequest.getProductNum()) {
            throw new PlatformRuntimeException("商品库存不足，请刷新后重新选择");
        }
        // 查询商品规格属性值信息
        EbProductAttrValue attrValue = BeanContextAware.getBeanByType(ProductAttrValueServiceImpl.class).queryByIdAndProductIdAndType(detailRequest.getAttrValueId(), detailRequest.getProductId(), ProductConstants.PRODUCT_TYPE_NORMAL);
        if (attrValue == null) {
            throw new PlatformRuntimeException("商品规格信息不存在，请刷新后重新选择");
        }
        if (attrValue.getStock() < detailRequest.getProductNum()) {
            throw new PlatformRuntimeException("商品规格库存不足，请刷新后重新选择");
        }
        EbMerchant merchant = BeanContextAware.getBeanByType(MerchantCache.class).get(product.getMerId());
        if (merchant.getIsSwitch().intValue() == 0) {
            throw new PlatformRuntimeException("商户已关闭，请重新选择商品");
        }
        PreMerchantOrderVo merchantOrderVo = new PreMerchantOrderVo();
        merchantOrderVo.setMerId(merchant.getId());
        merchantOrderVo.setMerName(merchant.getName());
        merchantOrderVo.setFreightFee(0d);
        merchantOrderVo.setCouponFee(0d);
        merchantOrderVo.setUserCouponId(0L);
        merchantOrderVo.setTakeTheirSwitch(merchant.getIsTakeTheir().intValue() == 1);
        PreOrderDetailVo detailVo = new PreOrderDetailVo();
        detailVo.setProductId(product.getId());
        detailVo.setProductName(product.getName());
        detailVo.setAttrValueId(attrValue.getId());
        detailVo.setSku(attrValue.getSku());

        // 2023-09-13，价格规格，是否用户输入'其他金额'
        if(StringUtils.isNotEmpty(detailRequest.getOtherMoney())){
            try {
                double otherMoneyValue = Double.parseDouble(detailRequest.getOtherMoney());
                detailVo.setPrice(NumberFormatUtils.scaleAccuracy2(otherMoneyValue));
                detailVo.setPayPrice(detailVo.getPrice());
            } catch (Exception ex){
                throw new PlatformRuntimeException("输入金额错误：" + detailRequest.getOtherMoney());
            }
        } else {
            // 正常规格对应的金额
            detailVo.setPrice(attrValue.getPrice());
            detailVo.setPayPrice(attrValue.getPrice());
        }

        detailVo.setPayNum(detailRequest.getProductNum());
        detailVo.setImage(StringUtils.isNotEmpty(attrValue.getImage()) ? cdnUrl+attrValue.getImage() : cdnUrl+product.getImage());
        detailVo.setVolume(attrValue.getVolume());
        detailVo.setWeight(attrValue.getWeight());
        detailVo.setTempId(product.getTempId());
//        detailVo.setSubBrokerageType(product.getIsSub() ? 1 : 2);
        detailVo.setSubBrokerageType(product.getIsSub());
        detailVo.setBrokerage(attrValue.getBrokerage().intValue());
        detailVo.setBrokerageTwo(attrValue.getBrokerageTwo().intValue());
        if (detailVo.getSubBrokerageType() == 2) {
            ArgumentsManager argumentsManager = BeanContextAware.getBeanByType(ArgumentsManager.class);
            String firstRatio = argumentsManager.getVariable(Constants.RETAIL_STORE_BROKERAGE_FIRST_RATIO).getStringValue();
            String secondRatio = argumentsManager.getVariable(Constants.RETAIL_STORE_BROKERAGE_SECOND_RATIO).getStringValue();
            detailVo.setBrokerage(StringUtils.isNotEmpty(firstRatio) ? Integer.parseInt(firstRatio) : 0);
            detailVo.setBrokerageTwo(StringUtils.isNotEmpty(secondRatio) ? Integer.parseInt(secondRatio) : 0);
        }
        detailVo.setProductType(ProductConstants.PRODUCT_TYPE_NORMAL);
        List<PreOrderDetailVo> infoList = new ArrayList<>(2);
        infoList.add(detailVo);
        merchantOrderVo.setOrderInfoList(infoList);
        return merchantOrderVo;
    }

    private static final PreMerchantOrderVo validatePreOrderVideo(PreOrderDetailRequest detailRequest) {
        // 暂时不实现该方法
        return new PreMerchantOrderVo();
    }
}
