package com.walker.cache;

import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;

import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * 缓存对象提供者
 * @author shikeying
 *
 * @param <T>
 */
public interface CacheProvider<T> extends InitializingBean, DisposableBean {

	/**
	 * 返回缓存对象
	 * @return
	 */
	Cache getCache();

	/**
	 * 返回缓存提供者的名字
	 * @return
	 */
	String getProviderName();

	/**
	 * 得到提供者泛型类型
	 * @return
	 */
	Class<?> getProviderType();

	/**
	 * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	 * 因为Cache接口不支持泛型，所以在<code>CacheProvider</code>
	 * 中加入了泛型方法，这些是直接访问缓存数据的方法，这可能有些</br>
	 * 不太合适，因为提供者只需要提供缓存对象就可以了，暂时先不增加</br>
	 * 新接口，直接在此扩充内容，随着演化，这个接口需要调整。
	 * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	 */

	/**
	 * 获得一个缓存数据
	 * @param key
	 * @return
	 */
	T getCacheData(String key);

	@Deprecated
	long getCacheCount();

	void removeCacheData(String key);

	void updateCacheData(String key, T data);

	void putCacheData(String key, T data);

	/**
	 * 缓存提供者提供有时间期限的数据存储。
	 * @param key
	 * @param data
	 * @param expiredSeconds 过期秒数
	 * @date 2022-11-06
	 */
	void putCacheData(String key, T data, long expiredSeconds);

	/**
	 * 重新加载缓存数据，该操作需要谨慎处理。
	 * <p>1.对于持久化 Redis 来说，重新加载意味着已有缓存磁盘数据，会全部被清空。至于会不会重新立即从数据库加载，需要看每个定义的需要。</p>
	 * @throws Exception
	 */
	void reload() throws Exception;

	void setLoadPage(boolean loadPage);

	boolean isLoadPage();

	void setPageSize(int pageSize);

	long getCreateTime();

	/**
	 * redis缓存需要传入更多参数
	 * @return
	 */
	Map<String, String> getCacheParam();

	Collection<Object> queryListLimit(int maxSize);

	long size();

	/**
	 * 存储写入集合缓存
	 * @param key 唯一ID
	 * @param data 数据集合
	 * @param expiredSeconds 失效时间（秒）
	 * @date 2023-06-13
	 */
	void putCacheList(String key, List<T> data, long expiredSeconds);

	/**
	 * 存储写入集合缓存
	 * @param key
	 * @param data
	 * @date 2023-06-13
	 */
	void putCacheList(String key, List<T> data);

	/**
	 * 向给定集合追加数据。
	 * @param key
	 * @param data
	 * @date 2023-06-13
	 */
	void putCacheListAppend(String key, T data);

	/**
	 * 获取缓存的集合
	 * @param key 唯一ID
	 * @return
	 * @date 2023-06-13
	 */
	List<T> getCacheList(String key);

	/**
	 * 删除给定key集合中的data数据，该方法只能删除集合中一个元素（值等于data）
	 * @param key
	 * @param data
	 * @date 2023-06-13
	 */
	void removeCacheList(String key, T data);

	/**
	 * 删除给定key中，整个集合数据。
	 * @param key
	 * @date 2023-08-03
	 */
	void removeCacheList(String key);
}
