package com.walker.cache;

import com.walker.cache.tree.CacheTree;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 应用系统缓存管理器默认实现
 * @author shikeying
 *
 */
public final class SimpleCacheManager {

	private static final transient Logger logger = LoggerFactory.getLogger(SimpleCacheManager.class);

	private static boolean initialized = false;

	private static final Object lock = new Object();

	private static final Map<Class<?>, CacheProvider<?>> cacheProvidersMap = new ConcurrentHashMap<Class<?>, CacheProvider<?>>();
	// 2023-01-04 能通过 CacheProvider.getProviderName() 获取到缓存对象
	private static final Map<String, CacheProvider<?>> cacheProvidersNameMap = new ConcurrentHashMap<String, CacheProvider<?>>();

	private SimpleCacheManager(){}

//	/**
//	 * 返回系统配置参数缓存对象提供者
//	 * @return
//	 */
//	@SuppressWarnings("unchecked")
//	public static final CacheProvider<PreferenceEntity> getPreferenceCacheProvider(){
//		CacheProvider<?> cp = cacheProvidersMap.get(PreferenceEntity.class);
//		if(cp == null) return null;
//		if(cp.getProviderType() == PreferenceEntity.class){
//			return (CacheProvider<PreferenceEntity>)cp;
//		} else
//			throw new RuntimeException("preferenceCache must be type of " + PreferenceEntity.class.getName());
//	}

	/**
	 * 根据缓存对象提供者名字，返回缓存对象。
	 * @param cacheProviderName
	 * @return
	 * @date 2023-01-04
	 */
	public static final CacheProvider<?> getCacheProvider(String cacheProviderName){
		CacheProvider<?> cp = cacheProvidersNameMap.get(cacheProviderName);
		if(cp == null)
			throw new NullPointerException("not found cacheProvider: " + cacheProviderName);
		return cp;
	}

	/**
	 * 返回缓存提供者
	 * @param clazz 对应实体类型，如：<code>PreferenceEntity.class</code>
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public static final <T> CacheProvider<T> getCacheProvider(Class<T> clazz){
		CacheProvider<?> cp = cacheProvidersMap.get(clazz);
		if(cp == null)
			throw new NullPointerException("not found cacheProvider: " + clazz.getName());
		return (CacheProvider<T>)cp;
	}

	/**
	 * 返回缓存树提供者对象
	 * @param clazz 对应实体类型，如：<code>CodeEntity.class</code>
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public static final <T> CacheTree<T> getCacheTreeProvider(Class<T> clazz){
		CacheProvider<?> cp = getCacheProvider(clazz);
		if(CacheTree.class.isAssignableFrom(cp.getClass())){
			return (CacheTree<T>)cp;
		}
		throw new UnsupportedOperationException(clazz.getName() + " is not a CacheTree!");
	}

	/**
	 * 返回系统定义的所有缓存对象。如果不存返回<code>null</code>
	 * @return
	 */
	@Deprecated
	public static final Collection<CacheProvider<?>> getCacheProviders(){
		if(cacheProvidersMap.size() == 0)
			return null;
		return cacheProvidersMap.values();
	}

	/**
	 * 获得所有缓存提供者对象集合，该方法替代：{@linkplain SimpleCacheManager#getCacheProviders()}
	 * @return
	 * @date 2023-08-27
	 */
	public static final List<CacheProvider<?>> getCacheProviderList(){
		List<CacheProvider<?>> data = new ArrayList<>();
		for(CacheProvider<?> cp : cacheProvidersNameMap.values()){
			data.add(cp);
		}
		return data;
	}

	public void setCacheProviders(List<CacheProvider<?>> cacheProviders){
		if(cacheProviders == null) return;
		if(!initialized){
			synchronized (lock) {
				try{
					// 在这里初始化所有缓存对象
					for(CacheProvider<?> cp : cacheProviders){
//						registeredCaches.put(cp.getProviderName(), cp.getCache());
						cacheProvidersMap.put(cp.getProviderType(), cp);
						cacheProvidersNameMap.put(cp.getProviderName(), cp);
					}
					initialized = true;

				} catch(Throwable cause){
					throw new Error("初始化系统缓存失败。", cause);
				}
			}
		}
	}

	/**
	 * 向缓存管理器中添加（注册）一个缓存对象
	 * @param cacheProvider
	 * @return 注册成功返回<code>true</code>
	 */
	public static final boolean addCacheProvider(CacheProvider<?> cacheProvider){
		assert (cacheProvider != null);
//		CacheProvider<?> cache = cacheProvidersMap.get(cacheProvider.getProviderType());
		if(!cacheProvidersMap.containsKey(cacheProvider.getProviderType())){
			cacheProvidersMap.put(cacheProvider.getProviderType(), cacheProvider);
			cacheProvidersNameMap.put(cacheProvider.getProviderName(), cacheProvider);
			return true;
		} else {
//			throw new IllegalStateException("cache already exist, can't add repeatly: " + cacheProvider.getProviderName());
			cacheProvidersNameMap.put(cacheProvider.getProviderName(), cacheProvider);
			logger.debug("已经存在了缓存对象，不能重复注册: " + cacheProvider.getProviderType() + ", cacheName=" + cacheProvider.getProviderName());
			return false;
		}
	}
}
