package com.walker.cache.tree;

import com.walker.infrastructure.utils.Assert;
import com.walker.infrastructure.utils.StringUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.TreeMap;

/**
 * 默认的缓存树节点实现
 * @author shikeying
 *
 */
public class DefaultCacheTreeNode implements CacheTreeNode {

	protected String key;
	protected String text;
	protected Object source;
	
	private String parentId;
	
	private int order = 0;
	
	protected TreeMap<String, CacheTreeNode> children = new TreeMap<String, CacheTreeNode>();
	
	private boolean checked = false;
	private boolean opened = false;
	
	private String icon = null;
	
	public DefaultCacheTreeNode(String key, String text
			, Object source, String parentId){
		assert(StringUtils.isNotEmpty(key));
		assert(StringUtils.isNotEmpty(text));
		this.key = key;
		this.text = text;
		this.source = source;
		if(StringUtils.isNotEmpty(parentId))
			this.parentId = parentId;
	}
	
	public DefaultCacheTreeNode(String key, String text
			, Object source, String parentId, int orderNum){
		this(key, text, source, parentId);
		this.order = orderNum;
	}
	
	@Override
	public String getKey() {
		return key;
	}

	@Override
	public String getText() {
		return text;
	}

	@Override
	public Object getSource() {
//		throw new UnsupportedOperationException("no source.");
		return this.source;
	}

	@Override
	public boolean hasChild() {
		return this.children.size() > 0;
	}

	@Override
	public int getChildrenSize() {
		return this.children.size();
	}

	@Override
	public Collection<CacheTreeNode> getChildren() {
		if(hasChild())
			return children.values();
		return null;
	}

	@Override
	public CacheTreeNode search(String key) {
		if(StringUtils.isEmpty(key)) return null;
		if(this.key.equals(key))
			return this;
		if(!hasChild())
			return null;
		
		CacheTreeNode child = null;
		for(Iterator<CacheTreeNode> i = children.values().iterator(); i.hasNext();){
			child = i.next().search(key);
			if(child != null)
				return child;
		}
		return null;
	}
	
	@Override
	public List<CacheTreeNode> searchLike(String keyLike){
//		throw new UnsupportedOperationException();
		if(StringUtils.isEmpty(keyLike)) return null;
		List<CacheTreeNode> list = new ArrayList<>(8);
		if(this.key.indexOf(keyLike) >= 0 
				|| this.text.indexOf(keyLike) >= 0){
			list.add(this);
		}
		if(hasChild()){
			List<CacheTreeNode> child = null;
			for(Iterator<CacheTreeNode> i = children.values().iterator(); i.hasNext();){
				child = i.next().searchLike(keyLike);
				if(child != null){
					list.addAll(child);
				}
			}
		}
		return list;
	}
	
	@Override
	public List<CacheTreeNode> searchLike(String[] keys){
//		throw new UnsupportedOperationException();
		if(keys == null) return null;
		List<CacheTreeNode> list = new ArrayList<>(8);
		
		if(this.containKeyLike(this.key, keys) 
				|| this.containKeyLike(this.text, keys)){
			list.add(this);
		}
		if(hasChild()){
			List<CacheTreeNode> child = null;
			for(Iterator<CacheTreeNode> i = children.values().iterator(); i.hasNext();){
				child = i.next().searchLike(keys);
				if(child != null){
					list.addAll(child);
				}
			}
		}
		return list;
	}
	
	/**
	 * 给定的关键词必须是并列存在的，有一个不匹配就返回失败
	 * @param content
	 * @param keys
	 * @return
	 */
	private boolean containKeyLike(String content, String[] keys){
		for(String k : keys){
			if(content.indexOf(k) < 0){
				return false;
			}
		}
		return true;
	}

	@Override
	public void addChild(CacheTreeNode node) {
		if(node != null){
			if(this.search(node.getKey()) == null){
				children.put(node.getKey(), node);
			} 
//			else
//				System.out.println("当前节点已经存在孩子对象，不再添加: " + node.getKey());
		}
	}

	@Override
	public String getParentId() {
		return this.parentId;
	}

	public String toString(){
		StringBuilder sb = new StringBuilder().append("{key = ").append(key)
				.append(", text = ").append(text)
				.append(", parentId = ").append(parentId)
				.append(", order = ").append(order);
		sb.append(",[");
		for(CacheTreeNode node : this.children.values()){
			sb.append(node.getKey());
			sb.append(":");
			sb.append(node.getText());
			sb.append(",");
		}
		sb.append("]");
		return sb.toString();
	}

	@Override
	public CacheTreeNode remove(String key) {
		if(StringUtils.isEmpty(key)) return null;
		if(!hasChild()){
			return null;
		}
		CacheTreeNode removed = children.remove(key);
		if(removed != null)
			return removed;
		for(Iterator<CacheTreeNode> i = children.values().iterator(); i.hasNext();){
			removed = i.next().remove(key);
			if(removed != null){
				return removed;
			}
		}
		return null;
	}

	@Override
	public void cloneProperties(CacheTreeNode node) {
		this.text = node.getText();
		this.source = node.getSource();
		this.order = node.getOrder();
	}

	@Override
	public int getOrder() {
		return this.order;
	}

	@Override
	public CacheTreeNode setOrder(int orderNum) {
		this.order = orderNum;
		return this;
	}

	@Override
	public int compareTo(CacheTreeNode o) {
		if(o == null) return 1;
		if(this == o) return 0;
		if(this.order > o.getOrder())
			return 1;
		else if(this.order < o.getOrder()){
			return -1;
		}
		return 0;
	}

	@Override
	public boolean isChecked() {
		return this.checked;
	}

	@Override
	public boolean isOpen() {
		return this.opened;
	}

	@Override
	public void setChecked(boolean bool) {
		this.checked = bool;
	}

	@Override
	public void setOpen(boolean bool) {
		this.opened = bool;
	}

	@Override
	public void setParentId(String parentId) {
		Assert.hasText(parentId);
		this.parentId = parentId;
	}

	@Override
	public String getIcon() {
		return icon;
	}

	@Override
	public void setIcon(String icon) {
		this.icon = icon;
	}

	@Override
	public CacheTreeNode copy() {
		CacheTreeNode newObject = new DefaultCacheTreeNode(key, text, source, parentId);
		newObject.setIcon(icon);
		return newObject.setOrder(this.order);
	}
}
