package com.walker.connector.util;

import com.walker.connector.Address;
import com.walker.connector.db.DamengConnector;
import com.walker.connector.db.MySqlConnector;
import com.walker.connector.db.OracleConnector;
import com.walker.connector.db.PostgresConnector;
import com.walker.connector.support.DatabaseConnector;
import com.walker.db.DatabaseType;
import com.walker.infrastructure.utils.StringUtils;
import org.springframework.jdbc.datasource.DataSourceTransactionManager;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.interceptor.TransactionProxyFactoryBean;

import javax.sql.DataSource;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

/**
 * 连接器工具类
 * @author shikeying
 * @date 2015年12月17日
 *
 */
public abstract class ConnectorUtils {

	/**
	 * 创建带事务管理的，数据库连接器对象。</p>
	 * 事务规则如下：
	 * <pre>
	 * "query*", "PROPAGATION_SUPPORTS,readOnly"
	 * "exec*", "PROPAGATION_REQUIRED, -Exception"
	 * </pre>
	 * @param connector
	 * @return
	 */
	public static final DatabaseConnector acquireTransactionProxyConnector(DatabaseConnector connector){
		PlatformTransactionManager transactionManager = new DataSourceTransactionManager(connector.getDataSource());

		// 定义事务规则
		Properties transactionAttributes = new Properties();
		transactionAttributes.put("query*", "PROPAGATION_SUPPORTS,readOnly");
		transactionAttributes.put("exec*", "PROPAGATION_REQUIRED, -Exception");

		TransactionProxyFactoryBean transactionProxy = new TransactionProxyFactoryBean();
		transactionProxy.setTransactionManager(transactionManager);
		transactionProxy.setTransactionAttributes(transactionAttributes);
		transactionProxy.setProxyTargetClass(true);
		transactionProxy.setTarget(connector);
		transactionProxy.afterPropertiesSet();
		return (DatabaseConnector)transactionProxy.getObject();
	}

	public static final DatabaseConnector createMySQLConnector(DataSource ds){
		return doCreateMySqLConnector(null, false, ds);
	}

	/**
	 * 创建MySQL的连接器
	 * @param address
	 * @return
	 */
	public static final DatabaseConnector createMySQLConnector(Address address){
		return doCreateMySqLConnector(address, false, null);
	}

	/**
	 * 创建MySQL的管理连接器，没有库名
	 * @param address
	 * @return
	 */
	public static final DatabaseConnector createMySQLManageConnector(Address address){
		return doCreateMySqLConnector(address, true, null);
	}

	private static DatabaseConnector doCreateMySqLConnector(Address address
			, boolean manageMode, DataSource ds){
		MySqlConnector dbConnector;

		if(ds != null){
			// 如果存在数据源，直接注入创建Connector对象
			dbConnector = new MySqlConnector(ds);
			return dbConnector;
		}

//		// 2023-05-04，使用系统配置统一数据源，否则会出现：业务用druid，而元数据用Hikari情况
//		if(JdbcInspector.getInstance() != null){
//			dbConnector = new MySqlConnector((DataSource) JdbcInspector.getInstance().getPrimaryDataSourceMeta());
//			return dbConnector;
//		}

		dbConnector = new MySqlConnector();
		dbConnector.setUrl(address.getUrl());
		dbConnector.setPort(address.getPort());
		//设置数据库名字，后面拼接字符集
//		dbConnector.setServiceName("walkersoft-share-marks?characterEncoding=UTF-8");
		if(manageMode){
			dbConnector.setManageMode();
		} else {
			if(StringUtils.isEmpty(address.getServiceName())){
				throw new IllegalArgumentException("未设置连接数据库名称：address.service = null.");
			}
			dbConnector.setServiceName(address.getServiceName()+"?characterEncoding=UTF-8");
		}

		Map<String, String> parameters = new HashMap<String, String>(5);
		parameters.put(DatabaseConnector.OPTION_USER, address.getAuthentication());
		parameters.put(DatabaseConnector.OPTION_PASSWORD, address.getCertification());
		if(address.getMaxActive() > 0){
			parameters.put(DatabaseConnector.OPTION_MAX_ACTIVE, String.valueOf(address.getMaxActive()));
		} else {
			parameters.put(DatabaseConnector.OPTION_MAX_ACTIVE, "6");
		}
		if(address.getMaxIdle() > 0){
			parameters.put(DatabaseConnector.OPTION_MAX_IDLE, String.valueOf(address.getMaxIdle()));
		} else {
			parameters.put(DatabaseConnector.OPTION_MAX_IDLE, "6");
		}
		if(address.getInitSize() > 0){
			parameters.put(DatabaseConnector.OPTION_INIT_SIZE, String.valueOf(address.getInitSize()));
		} else {
			parameters.put(DatabaseConnector.OPTION_INIT_SIZE, "3");
		}

		dbConnector.setParameters(parameters);
		dbConnector.initialize();
		return dbConnector;
	}

	public static final DatabaseConnector createOracleConnector(DataSource ds){
		return doCreateOracleConnector(null, false, ds);
	}
	public static final DatabaseConnector createOracleConnector(Address address){
		return doCreateOracleConnector(address, false, null);
	}
	public static final DatabaseConnector createOracleManageConnector(Address address){
		return doCreateOracleConnector(address, true, null);
	}

	/**
	 * 创建达梦数据库连接器对象，不是管理端，只能连特定数据库。
	 * @param address
	 * @return
	 * @date 2023-03-03
	 */
	public static final DatabaseConnector createDamengConnector(Address address){
		return doCreateDamengConnector(address, false, null);
	}
	public static final DatabaseConnector createDamengManageConnector(Address address){
		return doCreateDamengConnector(address, true, null);
	}

	private static DatabaseConnector doCreateDamengConnector(Address address, boolean manageMode, DataSource ds){
		DamengConnector dbConnector;
		if(ds != null){
			// 如果存在数据源，直接注入创建Connector对象
			dbConnector = new DamengConnector(ds);
			return dbConnector;
		}

//		// 2023-05-04，使用系统配置统一数据源，否则会出现：业务用druid，而元数据用Hikari情况
//		if(JdbcInspector.getInstance() != null){
//			dbConnector = new DamengConnector((DataSource) JdbcInspector.getInstance().getPrimaryDataSourceMeta());
//			return dbConnector;
//		}

		dbConnector = new DamengConnector();
		dbConnector.setUrl(address.getUrl());
		dbConnector.setPort(address.getPort());
		if(manageMode){
			dbConnector.setManageMode();
		} else {
			if(StringUtils.isEmpty(address.getServiceName())){
				throw new IllegalArgumentException("未设置连接数据库名称：address.service = null.");
			}
			dbConnector.setServiceName(address.getServiceName());
		}
		Map<String, String> parameters = new HashMap<String, String>(5);
		parameters.put(DatabaseConnector.OPTION_USER, address.getAuthentication());
		parameters.put(DatabaseConnector.OPTION_PASSWORD, address.getCertification());
		parameters.put(DatabaseConnector.OPTION_MAX_ACTIVE, "6");
		parameters.put(DatabaseConnector.OPTION_MAX_IDLE, "3");
		parameters.put(DatabaseConnector.OPTION_INIT_SIZE, "3");

		dbConnector.setParameters(parameters);
		dbConnector.initialize();
		return dbConnector;
	}

	private static DatabaseConnector doCreateOracleConnector(Address address
			, boolean manageMode, DataSource ds){
		OracleConnector dbConnector;
		if(ds != null){
			// 如果存在数据源，直接注入创建Connector对象
			dbConnector = new OracleConnector(ds);
			return dbConnector;
		}

//		// 2023-05-04，使用系统配置统一数据源，否则会出现：业务用druid，而元数据用Hikari情况
//		if(JdbcInspector.getInstance() != null){
//			dbConnector = new OracleConnector((DataSource) JdbcInspector.getInstance().getPrimaryDataSourceMeta());
//			return dbConnector;
//		}

		dbConnector = new OracleConnector();
		dbConnector.setUrl(address.getUrl());
		dbConnector.setPort(address.getPort());
		if(manageMode){
			dbConnector.setManageMode();
		} else {
			if(StringUtils.isEmpty(address.getServiceName())){
				throw new IllegalArgumentException("未设置连接数据库名称：address.service = null.");
			}
			dbConnector.setServiceName(address.getServiceName());
		}

		Map<String, String> parameters = new HashMap<String, String>(5);
		parameters.put(DatabaseConnector.OPTION_USER, address.getAuthentication());
		parameters.put(DatabaseConnector.OPTION_PASSWORD, address.getCertification());
		parameters.put(DatabaseConnector.OPTION_MAX_ACTIVE, "6");
		parameters.put(DatabaseConnector.OPTION_MAX_IDLE, "3");
		parameters.put(DatabaseConnector.OPTION_INIT_SIZE, "3");

		dbConnector.setParameters(parameters);
		dbConnector.initialize();
		return dbConnector;
	}

	public static final DatabaseConnector createPostgresConnector(Address address){
		return doCreatePostgresConnector(address, false);
	}

	public static final DatabaseConnector createPostgresManageConnector(Address address){
		return doCreatePostgresConnector(address, true);
	}

	private static DatabaseConnector doCreatePostgresConnector(Address address, boolean manageMode){
		PostgresConnector dbConnector = new PostgresConnector();

//		// 2023-05-04，使用系统配置统一数据源，否则会出现：业务用druid，而元数据用Hikari情况
//		if(JdbcInspector.getInstance() != null){
//			dbConnector = new PostgresConnector((DataSource) JdbcInspector.getInstance().getPrimaryDataSourceMeta());
//			return dbConnector;
//		}

		dbConnector.setUrl(address.getUrl());
		dbConnector.setPort(address.getPort());
		if(manageMode){
			dbConnector.setManageMode();
		} else {
			if(StringUtils.isEmpty(address.getServiceName())){
				throw new IllegalArgumentException("未设置连接数据库名称：address.service = null.");
			}
			dbConnector.setServiceName(address.getServiceName());
		}

		Map<String, String> parameters = new HashMap<String, String>(5);
		parameters.put(DatabaseConnector.OPTION_USER, address.getAuthentication());
		parameters.put(DatabaseConnector.OPTION_PASSWORD, address.getCertification());
		parameters.put(DatabaseConnector.OPTION_MAX_ACTIVE, "20");
//		parameters.put(DatabaseConnector.OPTION_MAX_IDLE, "3");
		parameters.put(DatabaseConnector.OPTION_INIT_SIZE, "3");

		dbConnector.setParameters(parameters);
		dbConnector.initialize();
		return dbConnector;
	}

	/**
	 * 返回一个数据库连接器对象
	 * @param addr 数据库连接地址
	 * @param manage 是否管理连接，管理连接没有数据库名
	 * @return
	 */
	public static final DatabaseConnector getDbConnectorByType(Address addr
			, boolean manage, DatabaseType databaseType){
		if(databaseType == DatabaseType.MYSQL){
			if(manage){
				return ConnectorUtils.createMySQLManageConnector(addr);
			} else {
				return ConnectorUtils.createMySQLConnector(addr);
			}
		} else if(databaseType == DatabaseType.POSTGRES){
			if(manage){
				return ConnectorUtils.createPostgresManageConnector(addr);
			} else {
				return ConnectorUtils.createPostgresConnector(addr);
			}
		} else if(databaseType == DatabaseType.ORACLE){
			if(manage){
				return ConnectorUtils.createOracleManageConnector(addr);
			} else {
				return ConnectorUtils.createOracleConnector(addr);
			}
		} else if (databaseType == DatabaseType.DAMENG) {
			if(manage){
				return ConnectorUtils.createDamengManageConnector(addr);
			} else {
				return ConnectorUtils.createDamengConnector(addr);
			}
		} else {
			throw new UnsupportedOperationException("未实现对其他数据库的支持：DatabaseConnector");
		}
	}

}
