package com.walker.connector.support;

import com.walker.api.client.JsonTransfer;
import com.walker.api.client.MapTransfer;
import com.walker.api.client.RequestData;
import com.walker.api.client.ResponseData;
import com.walker.connector.AbstractConnector;
import com.walker.remote.AbstractByteCoder;
import com.walker.remote.Constants;
import com.walker.remote.DoubleTrust;
import com.walker.remote.RemoteAccessor;
import com.walker.remote.util.HttpUtils;
import org.apache.hc.client5.http.cookie.CookieStore;

import java.util.Map;

/**
 * 移植旧代码。
 *
 * 支持 http 方式调用远程服务的连接器实现。
 * 该对象在中小型项目中可以方便使用，但在大型分布式项目中不要使用（请使用 openFeign）
 * @author 时克英
 * @date 2022-09-09
 */
public class HttpConnector extends AbstractConnector {

	private RemoteAccessor remoteAccessor = null;
	
	// 设置客户端证书验证信息
	private DoubleTrust clientTrust;
	
	public DoubleTrust getClientTrust() {
		return clientTrust;
	}

	public void setClientTrust(DoubleTrust clientTrust) {
		this.clientTrust = clientTrust;
	}

	@Override
	public Object invoke(Object... param) throws Exception {
		if(param.length < 2){
			throw new IllegalArgumentException("http调用参数缺失，至少2个：request, responseData");
		}
		if(param.length == 3){
			this.requestRemoteTextWithCookie(remoteAccessor
					, (RequestData)param[0]
					, (ResponseData<?>)param[1]
					, (CookieStore)param[2]);
		} else {
			this.requestRemoteText(remoteAccessor, (RequestData)param[0], (ResponseData<?>)param[1]);
		}
		return null;
	}

	@Override
	public void initialize() {
		super.initialize();
		if(remoteAccessor == null){
			remoteAccessor = this.getNewRemoteAccessor(getUrl());
		}
	}
	
	@Override
	public void destroy() {
		if(remoteAccessor != null){
			remoteAccessor = null;
		}
	}

	/**
	 * 返回一个新的远程调用对象
	 * @param remoteURL 访问地址
	 * @param contentCoder 编码、解码器，可以为<code>null</code>
	 * @return
	 */
	protected RemoteAccessor getNewRemoteAccessor(String remoteURL, AbstractByteCoder contentCoder){
		return HttpUtils.getInstance().getNewRemoteAccessor(remoteURL, contentCoder, clientTrust);
	}
	
	/**
	 * 返回一个新的远程调用对象
	 * @param remoteURL 访问地址
	 * @return
	 */
	protected RemoteAccessor getNewRemoteAccessor(String remoteURL){
		return HttpUtils.getInstance().getNewRemoteAccessor(remoteURL, null, clientTrust);
	}
	
	/**
	 * 异步请求远程服务，界面不阻塞。
	 * @param request 请求对象
//	 * @param taskCallback 用户界面回调实现
//	 * @param Context context 界面上下文对象
	 */
	public <T> void requestRemoteText(RemoteAccessor remoteAccessor
			, RequestData request, ResponseData<T> responseData){
		if(request == null){
			throw new IllegalArgumentException();
		}
		
		Class<?> requestDataClazz = request.getClass();
		
		if(MapTransfer.class.isAssignableFrom(requestDataClazz)){
			// 如果是实现Map参数方式
			MapTransfer mapTransfer = (MapTransfer)request;
			doRequestRemoteText(remoteAccessor, request.getRequestMethod()
					, mapTransfer.toMap(), null, responseData);
			
		} else if(JsonTransfer.class.isAssignableFrom(requestDataClazz)){
			// 实现的json传参方式
			JsonTransfer jsonTransfer = (JsonTransfer)request;
			doRequestRemoteText(remoteAccessor, request.getRequestMethod()
					, null, jsonTransfer.toJsonString(), responseData);
		}
	}
	
	/**
	 * 请求服务端文本消息
	 * @param methodName 方法名，服务端给定的调用方法名称，如：
	 * @param simpleData 提交的简单参数，如果存在只会处理该参数
	 * @param jsonParameter 发送给服务端的参数，json格式
	 */
	private <T> void doRequestRemoteText(RemoteAccessor remoteAccessor, String methodName
			, Map<String, String> simpleData
			, String jsonParameter, ResponseData<T> responseData){
		if(remoteAccessor == null){
			throw new IllegalStateException("remoteAccessor is required. 请先调用initialize()方法");
		}
		remoteAccessor.requestTextSync(methodName, simpleData, jsonParameter, responseData);
//		if(!responseData.getChannelStatus()){
//			logger.warn("请求返回错误状态: " + responseData.getChannelMessage());
//		}
	}
	
	public <T> void requestRemoteTextWithCookie(RemoteAccessor remoteAccessor
			, RequestData request, ResponseData<T> responseData
			, CookieStore cookieStore){
		if(request == null){
			throw new IllegalArgumentException();
		}
		
		Class<?> requestDataClazz = request.getClass();
		
		if(MapTransfer.class.isAssignableFrom(requestDataClazz)){
			// 如果是实现Map参数方式
			MapTransfer mapTransfer = (MapTransfer)request;
			doRequestRemoteTextWithCookie(remoteAccessor, request.getRequestMethod()
					, mapTransfer.toMap(), null, responseData, cookieStore);
			
		} else if(JsonTransfer.class.isAssignableFrom(requestDataClazz)){
			// 实现的json传参方式
			JsonTransfer jsonTransfer = (JsonTransfer)request;
			doRequestRemoteTextWithCookie(remoteAccessor, request.getRequestMethod()
					, null, jsonTransfer.toJsonString(), responseData, cookieStore);
		}
	}
	
	private <T> void doRequestRemoteTextWithCookie(RemoteAccessor remoteAccessor, String methodName
			, Map<String, String> simpleData
			, String jsonParameter, ResponseData<T> responseData
			, CookieStore cookieStore){
		if(remoteAccessor == null){
			throw new IllegalStateException("remoteAccessor is required. 请先调用initialize()方法");
		}
		remoteAccessor.requestTextSyncWithCookie(methodName, simpleData
				, jsonParameter, responseData, cookieStore);
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// 新添加方法，2020-07-06
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	public <T> void requestGet(String methodName
			, Map<String, String> simpleData, Map<String, String> header, ResponseData<T> responseData){
		this.checkRemoteAccessor();
		remoteAccessor.requestSync(methodName, simpleData, null
				, Constants.HTTP_METHOD_GET, Constants.APPLICATION_JSON, header, responseData);
	}
	
	public <T> void requestPost(String methodName
			, String jsonParameter, Map<String, String> header, ResponseData<T> responseData){
		this.checkRemoteAccessor();
		remoteAccessor.requestSyncPost(methodName, jsonParameter, header, responseData);
	}
	
	public <T> void requestPut(String methodName
			, String jsonParameter, Map<String, String> header, ResponseData<T> responseData){
		this.checkRemoteAccessor();
		remoteAccessor.requestSyncPut(methodName, jsonParameter, header, responseData);
	}
	
	public <T> void requestDelete(String methodName
			, String jsonParameter, Map<String, String> header, ResponseData<T> responseData){
		this.checkRemoteAccessor();
		remoteAccessor.requestSyncDelete(methodName, jsonParameter, header, responseData);
	}
	
	private void checkRemoteAccessor(){
		if(remoteAccessor == null){
			throw new IllegalStateException("remoteAccessor is required. 请先调用initialize()方法");
		}
	}
	
	public static void main(String[] args){
		HttpConnector connector = new HttpConnector();
		connector.setUrl("http://193.193.193.28:9200");
//		connector.setPort(9200);
		connector.initialize();
		
		// get
//		TextResponse response1 = new TextResponse();
//		Map<String, String> simpleData = new HashMap<>();
//		simpleData.put("analyzer", "ik_smart");
//		simpleData.put("text", "经核实，我司从未与腾讯公司或授权他人与腾讯公司就“老干妈”品牌签署《联合市场推广合作协议》");
//		connector.requestGet("/_analyze", simpleData, null, response1);
//		System.out.println(response1.getResultData());
		
		// post
		/*TextResponse response2 = new TextResponse();
		JSONObject postJson = new JSONObject();
		postJson.put("analyzer", "ik_smart");
		postJson.put("text", "经核实，我司从未与腾讯公司或授权他人与腾讯公司就“老干妈”品牌签署《联合市场推广合作协议》");
		connector.requestPost("/_analyze", postJson.toJSONString(), null, response2);
		System.out.println("2: post");
		System.out.println(response2.getResultData());
		
		// put
		long startTime = System.nanoTime();
		TextResponse response3 = new TextResponse();
		String jsonStr = "{\"query\":{\"multi_match\": {\"query\":\"免疫力 识别\",\"fields\":[ \"title\", \"attachment.content\" ]}}}";
		connector.requestPost("/demo/book/_search?pretty=true", jsonStr, null, response3);
		System.out.println("3: post(search): " + (System.nanoTime() - startTime));
		System.out.println(response2.getResultData());*/
	}
}
