package com.walker.dbmeta;

import com.walker.connector.Address;
import com.walker.connector.support.DatabaseConnector;
import com.walker.db.DatabaseException;
import com.walker.db.TableInfo;
import com.walker.db.page.GenericPager;

import java.util.List;
import java.util.Map;

/**
 * 数据库元数据引擎定义，仅获取数据库的元数据相关信息，与系统没有关系。</p>
 * 该对象为全局对象，通常创建单例使用。一个系统中可以创建多个，如：MySQL引擎、Oracle引擎等。</p>
 * 在数据采集系统中会存在许多数据源，因此建议实现类中增加数据连接的缓存器来管理这些链接。
 * @author shikeying
 * @date 2015年12月18日
 *
 */
public interface DatabaseMetaEngine {

	/**
	 * 返回表中字段对象集合
	 * @param address
	 * @param tableName
	 * @return
	 */
	public List<FieldInfo> getFieldsObject(Address address, String tableName);
	
	/**
	 * 返回表中所有字段集合
	 * @param address 连接的数据库地址
	 * @param tableName 查询的表名
	 * @return
	 */
	List<String> getFields(Address address, String tableName);
	
	/**
	 * 返回该地址数据库中存在的表数量
	 * @param address
	 * @return
	 */
	int getTableSize(Address address);
	
	/**
	 * 返回给定表集合中，每个表的概要数据量（行数）。<br>
	 * 该数据是从数据库元数据表中查询的值，只是一个参考结果。
	 * @param address
	 * @param tableNameList
	 * @return map&lt;tableName, rows&gt;
	 */
	Map<String, TableInfo> getTableRows(Address address, List<String> tableNameList);
	
	/**
	 * 返回给定的表当前记录数
	 * @param address
	 * @param tableName
	 * @return
	 */
	long getTableRow(Address address, String tableName);
	
	List<Map<String, Object>> loadTableDatas(Address address, String tableName, String sql);
	
	List<String> getTableNamesByLike(Address address, String tableNameLike);
	
	/**
	 * 分页查询给定名字前缀的表集合（只包含表名）
	 * @param address
	 * @param tableNameLike
	 * @return
	 */
	GenericPager<TableInfo> queryPageTableNamesByLike(Address address, String tableNameLike
//			, int pageIndex, int pageSize
			);
	
	/**
	 * 初始化，创建该对象后必须首先调用该方法。
	 */
	void initialize();
	
	/**
	 * 使用完该对象，必须调用该方法销毁
	 */
	void destroy();
	
	/**
	 * 动态创建表结果，可能是关系数据库，也可能是<code>NoSQL</code>数据库类型。</p>
	 * 给定一个数据集合，从中分析数据类型，生成简单的存储表。<br>
	 * 因为这个是在数据采集时使用，所以目前仅支持三种数据类型：string|long|double
	 * @param address 目的地址
	 * @param datas 给定的数据集合
	 * @param dataVersionField 数据版本字段名称，该字段用来增量采集数据对比，通常是数值类型。
	 * @param tableName 要创建的表名称
	 * @throws DatabaseException 创建失败，抛出异常
	 */
	void createTableDynamic(Address address
			, List<Map<String, Object>> datas, String dataVersionField, String tableName) throws DatabaseException;

	/**
	 * 直接设置连接器对象，不再调用方法：initialize()
	 * @param connector
	 */
	void setConnector(DatabaseConnector connector);

	DatabaseConnector getConnector();
}
