package com.walker.file.fs;

import com.walker.file.AbstractFileOperateEngine;
import com.walker.file.DefaultFileInfo;
import com.walker.file.FileInfo;
import com.walker.file.FileOperateException;
import com.walker.file.FileStoreType;
import com.walker.infrastructure.utils.FileCopyUtils;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;

public abstract class FileSystemOperateEngine extends AbstractFileOperateEngine {

    @Override
    protected FileInfo acquireFileInfo(String fileName) {
        return new DefaultFileInfo();
    }

    @Override
    protected void executeUpload(InputStream inputStream, FileInfo fileInfo) throws FileOperateException {
        String directory = this.generateSaveFolder(false);
        StringBuilder url = new StringBuilder(directory).append(this.getFileName(fileInfo));

        String outputFolder = null;
        String outputFileName = null;
        if(!this.getFileRoot().endsWith("/")){
            outputFileName = this.getFileRoot() + "/" + url.toString();
            outputFolder = this.getFileRoot() + "/" + directory;
        } else {
            outputFileName = this.getFileRoot() + url.toString();
            outputFolder = this.getFileRoot() + directory;
        }
        logger.debug("保存文件名称: " + outputFileName);
        try {
            File outputFolderFile = new File(outputFolder);
            if(!outputFolderFile.exists()){
                outputFolderFile.mkdirs();
            }
            BufferedOutputStream outputStream = new BufferedOutputStream(new FileOutputStream(outputFileName));
            long fileSize = FileCopyUtils.copy(inputStream, outputStream);
            ((DefaultFileInfo)fileInfo).setFileSize(fileSize);
            ((DefaultFileInfo)fileInfo).setUrl(url.toString());

        } catch (IOException e) {
            throw new FileOperateException("保存文件到系统磁盘错误:" + e.getMessage() + ", id=" + fileInfo.getId(), e);
        }
    }

    @Override
    protected byte[] executeDownload(FileInfo fileInfo) throws FileOperateException{
        String outputFileName = null;
        if(!this.getFileRoot().endsWith("/")){
            outputFileName = this.getFileRoot() + "/" + fileInfo.getUrl();
        } else {
            outputFileName = this.getFileRoot() + fileInfo.getUrl();
        }
        File outputFile = new File(outputFileName);
        if(!outputFile.exists()){
            throw new FileOperateException("下载文件不存在，fileId=" + fileInfo.getId() + ", url=" + fileInfo.getUrl(), null);
        }
        try {
            BufferedInputStream inputStream = new BufferedInputStream(new FileInputStream(outputFile));
            return FileCopyUtils.copyToByteArray(inputStream);
        } catch (Exception e) {
            throw new FileOperateException("下载文件流出错, error=" + e.getMessage(), e);
        }
    }

    @Override
    public FileStoreType getFileStoreType() {
        return FileStoreType.FileSystem;
    }
}
