package com.walker.common.util;

import com.fasterxml.jackson.databind.node.ObjectNode;
import com.walker.infrastructure.ApplicationRuntimeException;
import com.walker.infrastructure.utils.JsonUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.BeanWrapper;
import org.springframework.beans.BeanWrapperImpl;
import org.springframework.beans.NotReadablePropertyException;

import java.beans.BeanInfo;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 转换工具类
 *
 * @author 时克英
 */
public class ConvertUtil {
    private static final Logger logger = LoggerFactory.getLogger(ConvertUtil.class);

    /**
     * 根据对象属性的类型转换value为目标对象
     *
     * @param value 输入数据
     * @param clazz 类型
     * @return 转换后的对象
     */
    public static <T> Object convert(String value, Class<T> clazz) {
        try {
            String typeName = clazz.getCanonicalName();
            if (typeName.equals("java.lang.String")) {
                return value;
            }
            if (value == null || value.equals("") || value.equals("null") || value.equals("undefined")) {
                return null;
            } else if (typeName.equals("boolean") || typeName.equals("java.lang.Boolean")) {
                return Boolean.parseBoolean(value);
            } else if (typeName.equals("int") || typeName.equals("java.lang.Integer")) {
                return Integer.parseInt(value);
            } else if (typeName.equals("short") || typeName.equals("java.lang.Short")) {
                return Short.parseShort(value);
            } else if (typeName.equals("long") || typeName.equals("java.lang.Long")) {
                return Long.parseLong(value);
            } else if (typeName.equals("double") || typeName.equals("java.lang.Double")) {
                return Double.parseDouble(value);
            } else if (typeName.equals("float") || typeName.equals("java.lang.Float")) {
                return Float.parseFloat(value);
            } else if (typeName.equals("java.math.BigInteger")) {
                return new java.math.BigInteger(value);
            } else if (typeName.equals("java.math.BigDecimal")) {
                return new BigDecimal(value);
            } else if (typeName.equals("java.sql.Date")) {
                return java.sql.Date.valueOf(value.substring(0, 10));    //yyyy-mm-dd
            } else if (typeName.equals("java.sql.Time")) {
                return java.sql.Time.valueOf(value);
            } else if (typeName.equals("java.sql.Timestamp")) {
                if (value.trim().length() == 10) {
                    return java.sql.Timestamp.valueOf(value.trim() + " 00:00:00");
                }
                return java.sql.Timestamp.valueOf(value);
            }
            return null;
        } catch (Exception e) {
            logger.error("convert exception : value -> {}, type -> {}", new Object[]{value, clazz});
            logger.error(e.getMessage(), e);
            return null;
        }
    }

    /**
     * 根据对象属性的类型转换value为目标对象
     *
     * @param value 输入数据
     * @param clazz 类型
     * @return 转换后的对象
     */
    public static <T> Object convert(Object value, Class<T> clazz) {
        return convert(createString(value), clazz);
    }

    /**
     * 将map的key转成小写
     *
     * @param sourceMap
     * @return
     */
    public static Map<String, Object> convertKeyToLowerCase(Map<String, Object> sourceMap) {
        if (sourceMap == null) {
            return null;
        } else {
            try {
                Map mapDest = sourceMap.getClass().newInstance();
                for (Map.Entry<String, Object> entry : sourceMap.entrySet()) {
                    mapDest.put(entry.getKey().toLowerCase(), entry.getValue());
                }
                return mapDest;
            } catch (Exception e) {
                throw new ApplicationRuntimeException(e.getMessage(), e);
            }
        }
    }

    /**
     * obj转换为 int， 默认值为Integer.MIN_VALUE
     *
     * @param obj 待转换对象
     * @return int
     */
    public static int toInt(Object obj) {
        if (obj == null) {
            return Integer.MIN_VALUE;
        } else {
            return createInteger(obj);
        }
    }

    /**
     * obj转换为 int， 默认值为defaultValue
     *
     * @param obj 待转换对象
     * @return int
     */
    public static int toInt(Object obj, int defaultValue) {
        if (obj == null) {
            return defaultValue;
        } else {
            return createInteger(obj);
        }
    }

    /**
     * obj转换为 long， 默认值为Long.MIN_VALUE
     *
     * @param obj 待转换对象
     * @return long
     */
    public static long toLong(Object obj) {
        if (obj == null) {
            return Long.MIN_VALUE;
        } else {
            return createLong(obj);
        }
    }

    /**
     * obj转换为 long， 默认值为defaultValue
     *
     * @param obj 待转换对象
     * @return long
     */
    public static long toLong(Object obj, long defaultValue) {
        if (obj == null) {
            return defaultValue;
        } else {
            return createLong(obj);
        }
    }

    /**
     * obj转换为 double， 默认值为Double.MIN_VALUE
     *
     * @param obj 待转换对象
     * @return double
     */
    public static double toDouble(Object obj) {
        if (obj == null) {
            return Double.MIN_VALUE;
        } else {
            return createDouble(obj);
        }
    }

    /**
     * obj转换为 double， 默认值为defaultValue
     *
     * @param obj 待转换对象
     * @return double
     */
    public static double toDouble(Object obj, double defaultValue) {
        if (obj == null) {
            return defaultValue;
        } else {
            return createDouble(obj);
        }
    }

    /**
     * obj转换为 boolean， 默认值为false
     *
     * @param obj 待转换对象
     * @return boolean
     */
    public static boolean toBoolean(Object obj) {
        return obj != null && createBoolean(obj);
    }

    /**
     * obj转换为 boolean， 默认值为defaultValue
     *
     * @param obj 待转换对象
     * @return boolean
     */
    public static boolean toBoolean(Object obj, boolean defaultValue) {
        if (obj == null) {
            return defaultValue;
        } else {
            return createBoolean(obj);
        }
    }

    /**
     * 将obj转换为String
     *
     * @param obj 待转换对象
     * @return string
     */
    public static String createString(Object obj) {
        if (obj == null) {
            return null;
        } else {
            if (obj instanceof String) {
                return (String) obj;
            } else {
                return obj.toString();
            }
        }
    }

    /**
     * 将obj转换为Integer
     *
     * @param obj 待转换对象
     * @return Integer
     */
    public static Integer createInteger(Object obj) {
        if (obj == null) {
            return null;
        } else {
            if (obj instanceof Integer) {
                return (Integer) obj;
            } else {
                return Integer.valueOf(createString(obj));
            }
        }
    }

    /**
     * 将obj转换为Long
     *
     * @param obj 待转换对象
     * @return Long
     */
    public static Long createLong(Object obj) {
        if (obj == null) {
            return null;
        } else {
            if (obj instanceof Long) {
                return (Long) obj;
            } else {
                return Long.valueOf(createString(obj));
            }
        }
    }

    /**
     * 将obj转换为Double
     *
     * @param obj 待转换对象
     * @return Double
     */
    public static Double createDouble(Object obj) {
        if (obj == null) {
            return null;
        } else {
            if (obj instanceof Double) {
                return (Double) obj;
            } else {
                return Double.valueOf(createString(obj));
            }
        }
    }

    /**
     * 将obj转换为BigDecimal
     *
     * @param obj 待转换对象
     * @return BigDecimal
     */
    public static BigDecimal createBigDecimal(Object obj) {
        if (obj == null) {
            return null;
        } else {
            if (obj instanceof BigDecimal) {
                return (BigDecimal) obj;
            } else {
                return new BigDecimal(createString(obj));
            }
        }
    }

    /**
     * 将obj转换为Boolean
     *
     * @param obj 待转换对象
     * @return Boolean
     */
    public static Boolean createBoolean(Object obj) {
        if (obj == null) {
            return null;
        } else {
            if (obj instanceof Boolean) {
                return (Boolean) obj;
            } else {
                return Boolean.parseBoolean(createString(obj));
            }
        }
    }

    /**
     * 将源对象转化为targetClass类型的目标对象
     *
     * @param source      源对象
     * @param targetClass 目标对象
     * @param <T>         返回对象泛型
     * @return 转化后的对象
     */
    public static <T> T toObject(Object source, Class<T> targetClass) {
        if (source == null) {
            return null;
        }
        try {
            String typeName = targetClass.getCanonicalName();
            if (typeName.equals("java.util.Map") || typeName.equals("java.util.HashMap")) {
                Map t = new HashMap();
                copyProperties(source, t);
                return (T) t;
            } else if (typeName.equals("java.util.LinkedHashMap")) {
                Map t = new LinkedHashMap();
                copyProperties(source, t);
                return (T) t;
            } else {
                T t = targetClass.newInstance();
                copyProperties(source, t);
                return t;
            }
        } catch (Exception e) {
            throw new ApplicationRuntimeException(e);
        }
    }

    /**
     * 将源对象的属性拷贝到目标对象
     *
     * @param source 源对象
     * @param target 目标对象
     */
    public static void copyProperties(Object source, Object target) {
        if (source == null || target == null) {
            return;
        }
        if (source instanceof Map) {
            PropertyDescriptor[] propertyDescriptors = BeanUtils.getPropertyDescriptors(target.getClass());
            Map map = (Map) source;
            try {
                for (PropertyDescriptor descriptor : propertyDescriptors) {
                    if (map.containsKey(descriptor.getName())) {
                        Method writeMethod = descriptor.getWriteMethod();
                        Object value = map.get(descriptor.getName());
                        writeMethod.invoke(target, convert(value, writeMethod.getParameterTypes()[0]));
                    }
                }
            } catch (Exception e) {
                throw new ApplicationRuntimeException(e);
            }
        } else if (target instanceof Map) {
            try {
                PropertyDescriptor[] propertyDescriptors = BeanUtils.getPropertyDescriptors(source.getClass());
                Map<String, Object> map = (Map) target;
                for (PropertyDescriptor descriptor : propertyDescriptors) {
                    Method readMethod = descriptor.getReadMethod();
                    map.put(descriptor.getName(), readMethod.invoke(source, new Object[]{}));
                }
            } catch (Exception e) {
                throw new ApplicationRuntimeException(e);
            }
        } else {
            BeanUtils.copyProperties(source, target);
        }
    }


    /**
     * 将源对象的属性拷贝到目标对象,忽略为空的属性
     *
     * @param source 源对象
     * @param target 目标对象
     */
    public static void copyPropertiesIgnoreNull(Object source, Object target) {
        if (source == null || target == null) {
            return;
        }
        if (source instanceof Map) {
            PropertyDescriptor[] propertyDescriptors = BeanUtils.getPropertyDescriptors(target.getClass());
            Map map = (Map) source;
            try {
                for (PropertyDescriptor descriptor : propertyDescriptors) {
                    if (map.containsKey(descriptor.getName())) {
                        Method writeMethod = descriptor.getWriteMethod();
                        Object value = map.get(descriptor.getName());
                        writeMethod.invoke(target, convert(value, writeMethod.getParameterTypes()[0]));
                    }
                }
            } catch (Exception e) {
                throw new ApplicationRuntimeException(e);
            }
        } else if (target instanceof Map) {
            try {
                PropertyDescriptor[] propertyDescriptors = BeanUtils.getPropertyDescriptors(source.getClass());
                Map<String, Object> map = (Map) target;
                for (PropertyDescriptor descriptor : propertyDescriptors) {
                    Method readMethod = descriptor.getReadMethod();
                    map.put(descriptor.getName(), readMethod.invoke(source, new Object[]{}));
                }
            } catch (Exception e) {
                throw new ApplicationRuntimeException(e);
            }
        } else {
            BeanUtils.copyProperties(source, target, getNullPropertyNames(source));
        }
    }

    /**
     * 获取为空的属性名
     *
     * @param source 原对象
     * @return 源对象为空的属性名
     */
    private static String[] getNullPropertyNames(Object source) {
        final BeanWrapper src = new BeanWrapperImpl(source);
        PropertyDescriptor[] pds = src.getPropertyDescriptors();

        Set<String> emptyNames = new HashSet<>();
        for (PropertyDescriptor pd : pds) {
            Object srcValue = null;
            try {
                srcValue = src.getPropertyValue(pd.getName());
            } catch (NotReadablePropertyException e) {
                logger.error(pd.getName() + "属性获取出错", e);
            }
            if (srcValue == null) {
                emptyNames.add(pd.getName());
            }
        }
        String[] result = new String[emptyNames.size()];
        return emptyNames.toArray(result);
    }

    /**
     * 将源对象列表转化为目标对象列表
     *
     * @param list        源对象列表
     * @param targetClass 目标对象类型
     * @param <T>         返回对象泛型
     * @return
     */
    public static <T> List<T> toList(List<?> list, Class<T> targetClass) {
        if (list == null) {
            return null;
        }
        List<T> result = new ArrayList<>();
        for (Object obj : list) {
            result.add(toObject(obj, targetClass));
        }
        return result;
    }

    /**
     * 将list 按 chunkSize 切片
     *
     * @param list      待切片数据
     * @param chunkSize 分片大小
     * @param <T>       返回对象泛型
     * @return
     */
    public static <T> List<List<T>> partition(List<T> list, int chunkSize) {
        if (list == null) {
            return null;
        }
        if (chunkSize <= 0) {
            throw new ApplicationRuntimeException("chunkSize 必须大于0.");
        }
        List<List<T>> result = new ArrayList<>();
        int index = 0;
        while (index < list.size()) {
            int nextInc = Math.min(list.size() - index, chunkSize);
            result.add(list.subList(index, index + nextInc));
            index += nextInc;
        }
        return result;
    }

    /**
     * 将多个业务信息转换为json数组
     *
     * @param datas "项目编号","BH001","项目预算","500w"
     * @return [{"name":"项目编号","value":"BH001"},{"name":"项目预算","value":"500w"}]
     * @throws Exception
     */
    public static String convert(String... datas) throws Exception {
        if (datas == null || datas.length < 0 || datas.length % 2 == 1) {
            throw new ApplicationRuntimeException("请确保参数数量大于0且为偶数！");
        }
//        String jsonData = null;
        int dataSize = datas.length / 2;
//        JSONArray jsonArray = new JSONArray();
//        for (int i = 0; i < dataSize; i++) {
//            int dataIndex = i * 2;
//            JSONObject jsonObject = new JSONObject();
//            jsonObject.put("name", datas[dataIndex]);
//            jsonObject.put("value", datas[dataIndex + 1]);
//            jsonArray.put(jsonObject);
//        }
//        jsonData = jsonArray.toString();
        List<ObjectNode> resultList = new ArrayList<>(4);
        int dataIndex = -1;
        Map<String, Object> map = null;
        for (int i = 0; i < dataSize; i++){
            dataIndex = i * 2;
            map = new HashMap<>(8);
            map.put("name", datas[dataIndex]);
            map.put("value", datas[dataIndex + 1]);
            resultList.add(JsonUtils.mapToObjectNode(map));
        }
        return JsonUtils.toJsonArray(resultList);
    }

    /**
     * 将多个业务信息转换为json数组
     *
     * @param datas "xmid","1001",0,"项目编号","BH001",1,"项目预算","500w",1
     * @return [{"name":"xmid","value":"1001","show":0},{"name":"项目编号","value":"BH001","show":1},{"name":"项目预算","value":"500w","show":1}]
     * @throws Exception
     */
    public static String convert3(String... datas) throws Exception {
        if (datas == null || datas.length < 0 || datas.length % 3 != 0) {
            throw new ApplicationRuntimeException("请确保参数数量大于0且为3的倍数！");
        }
        String jsonData = null;
        int dataSize = datas.length / 3;
//        JSONArray jsonArray = new JSONArray();
//        for (int i = 0; i < dataSize; i++) {
//            int dataIndex = i * 3;
//            JSONObject jsonObject = new JSONObject();
//            jsonObject.put("name", datas[dataIndex]);
//            jsonObject.put("value", datas[dataIndex + 1]);
//            jsonObject.put("show", datas[dataIndex + 2]);
//            jsonArray.put(jsonObject);
//        }
//        jsonData = jsonArray.toString();
        List<ObjectNode> resultList = new ArrayList<>(4);
        int dataIndex = -1;
        Map<String, Object> map = null;
        for (int i = 0; i < dataSize; i++){
            dataIndex = i * 3;
            map = new HashMap<>(8);
            map.put("name", datas[dataIndex]);
            map.put("value", datas[dataIndex + 1]);
            map.put("show", datas[dataIndex + 2]);
            resultList.add(JsonUtils.mapToObjectNode(map));
        }
        return JsonUtils.toJsonArray(resultList);
    }

    /**
     * 将bean的Stirng类型赋值为""
     *
     * @param bean
     * @param <T>
     * @return
     */
    public static <T> T convertStrPropBlank(T bean) {
        if (bean != null) {
            PropertyDescriptor[] propertyDescriptors = BeanUtils.getPropertyDescriptors(bean.getClass());
            for (PropertyDescriptor propertyDescriptor : propertyDescriptors) {
                if (propertyDescriptor.getPropertyType().getName().equals("java.lang.String")) {
                    try {
                        if (propertyDescriptor.getReadMethod().invoke(bean) == null) {
                            propertyDescriptor.getWriteMethod().invoke(bean, "");
                        }
                    } catch (IllegalAccessException e) {
                        logger.error(e.getMessage(), e);
                    } catch (InvocationTargetException e) {
                        logger.error(e.getMessage(), e);
                    }
                }
            }
        }
        return bean;
    }

    /**
     * 把JavaBean转化为map
     *
     * @param bean
     * @return
     */
    public static Map<String, Object> beanTomap(Object bean) throws Exception {
        Map<String, Object> map = new HashMap<>();
        //获取JavaBean的描述器
        BeanInfo b = Introspector.getBeanInfo(bean.getClass(), Object.class);
        //获取属性描述器
        PropertyDescriptor[] pds = b.getPropertyDescriptors();
        //对属性迭代
        for (PropertyDescriptor pd : pds) {
            //属性名称
            String propertyName = pd.getName();
            //属性值,用getter方法获取
            Method m = pd.getReadMethod();
            Object properValue = m.invoke(bean);//用对象执行getter方法获得属性值

            //把属性名-属性值 存到Map中
            map.put(propertyName, properValue);
        }
        return map;
    }

    /**
     * 把Map转化为JavaBean
     *
     * @param map
     * @return
     */
    /*public static <T> T mapTobean(Map<String,Object> map,Class<T> clz) throws Exception{
        //创建一个需要转换为的类型的对象
        T obj = clz.newInstance();
        //从Map中获取和属性名称一样的值，把值设置给对象(setter方法)

        //得到属性的描述器
        BeanInfo b = Introspector.getBeanInfo(clz,Object.class);
        PropertyDescriptor[] pds = b.getPropertyDescriptors();
        for (PropertyDescriptor pd : pds) {
            //得到属性的setter方法
            Method setter = pd.getWriteMethod();
            //得到key名字和属性名字相同的value设置给属性
            setter.invoke(obj, map.get(pd.getName()));
        }
        return obj;
    }*/
    public static <T> T mapToBean(Map<String, Object> map, Class<?> clazz) throws Exception {
        Object obj = clazz.newInstance();
        if (map != null && !map.isEmpty() && map.size() > 0) {
            for (Map.Entry<String, Object> entry : map.entrySet()) {
                String propertyName = entry.getKey();    // 属性名
                Object value = entry.getValue();        // 属性值
                String setMethodName = "set" + propertyName.substring(0, 1).toUpperCase() + propertyName.substring(1);
                Field field = getClassField(clazz, propertyName);    //获取和map的key匹配的属性名称
                if (field == null) {
                    continue;
                }
                Class<?> fieldTypeClass = field.getType();
                value = convertValType(value, fieldTypeClass);
                try {
                    clazz.getMethod(setMethodName, field.getType()).invoke(obj, value);
                } catch (NoSuchMethodException e) {
                    logger.error(e.getMessage(), e);
                }
            }
        }
        return (T) obj;
    }

    /**
     * 根据给定对象类匹配对象中的特定字段
     *
     * @param clazz
     * @param fieldName
     * @return
     */
    private static Field getClassField(Class<?> clazz, String fieldName) {
        if (Object.class.getName().equals(clazz.getName())) {
            return null;
        }
        Field[] declaredFields = clazz.getDeclaredFields();
        for (Field field : declaredFields) {
            if (field.getName().equals(fieldName)) {
                return field;
            }
        }
        Class<?> superClass = clazz.getSuperclass();    //如果该类还有父类，将父类对象中的字段也取出
        if (superClass != null) {                        //递归获取
            return getClassField(superClass, fieldName);
        }
        return null;
    }

    /**
     * 将map的value值转为实体类中字段类型匹配的方法
     *
     * @param value
     * @param fieldTypeClass
     * @return
     */
    private static Object convertValType(Object value, Class<?> fieldTypeClass) {
        Object retVal = null;

        if (Long.class.getName().equals(fieldTypeClass.getName())
                || long.class.getName().equals(fieldTypeClass.getName())) {
            retVal = Long.parseLong(value.toString());
        } else if (Integer.class.getName().equals(fieldTypeClass.getName())
                || int.class.getName().equals(fieldTypeClass.getName())) {
            retVal = Integer.parseInt(value.toString());
        } else if (Float.class.getName().equals(fieldTypeClass.getName())
                || float.class.getName().equals(fieldTypeClass.getName())) {
            retVal = Float.parseFloat(value.toString());
        } else if (Double.class.getName().equals(fieldTypeClass.getName())
                || double.class.getName().equals(fieldTypeClass.getName())) {
            retVal = Double.parseDouble(value.toString());
        } else {
            retVal = value;
        }
        return retVal;
    }
}
