package com.walker.db.page;

import java.util.List;

import com.walker.db.page.Pager;

public abstract class AbstractPager implements Pager {

	/* 当前给定的页码索引，如：1 = 第一页 */
	private int pageIndex = DEFAULT_PAGE_INDEX;
	
	/* 每页设置的数据条数，即：每页显示几条 */
	private int pageSize  = DEFAULT_PAGE_SIZE;
	
	private int pageCount = DEFAULT_PAGE_COUNT;
	
	private long totalRows = DEFAULT_TOTAL_ROWS;
	
	public AbstractPager(int pageIndex, int pageSize, long totalRows){
		setPageIndex(pageIndex);
		setPageSize(pageSize);
		if(totalRows > 0)
			setTotalRows(totalRows);
	}
	
	protected abstract Class<?> buildData();
	
	@Override
	public boolean isEmpty(){
		return totalRows == 0;
	}
	
	@Override
	public int getPageCount() {
		return pageCount;
	}

	@Override
	public long getTotalRows() {
		return totalRows;
	}

	@Override
	public void setTotalRows(final long totalRows){
		if(totalRows < 0)
			throw new IllegalArgumentException("totalRows must more than 0.");
		if(totalRows == 0) return;
		
//		if(totalRows == 0 && this.pageIndex != 1){
//			throw new IndexOutOfBoundsException("Page index out of range: " + this.pageIndex);
//		} 
//		else if(this.pageIndex > totalRows){
//			throw new IndexOutOfBoundsException("Page index out of range: " + this.pageIndex);
//		}
		this.totalRows = totalRows;
		this.pageCount = (int)(totalRows / this.pageSize + (totalRows % this.pageSize == 0 ? 0 : 1));
	}

	@Override
	public int getPageIndex() {
		return pageIndex;
	}

	public void setPageIndex(int pageIndex) {
		if(this.pageIndex > DEFAULT_PAGE_INDEX && pageIndex >= DEFAULT_PAGE_INDEX){
			throw new IllegalStateException("can't invoke method of setPageIndex() repeatedly.");
		}
		if (pageIndex < DEFAULT_PAGE_INDEX)
		      pageIndex = DEFAULT_PAGE_INDEX;
		this.pageIndex = pageIndex;
	}

	@Override
	public int getPageSize() {
		return pageSize;
	}

	public void setPageSize(int pageSize) {
		if(this.pageSize > DEFAULT_PAGE_SIZE 
				&& pageSize >= 0){
			throw new IllegalStateException("can't invoke method of setPageSize() repeatedly."
					+ ", this.pageSize = " + this.pageSize
					+ ", pageSize = " + pageSize);
		}
		if (pageSize < DEFAULT_PAGE_SIZE)
		      pageSize = DEFAULT_PAGE_SIZE;
		this.pageSize = pageSize;
	}
	
	/**
	 * 返回分页对象中的第一条记录的索引值
	 * @return
	 */
	@Override
	public long getFirstRowIndexInPage(){
		return (this.pageIndex - 1) * this.pageSize;
	}
	
	/**
	 * 返回本页最后一条记录索引值
	 * @return
	 */
	@Override
	public long getEndRowIndexPage(){
		long endDataIndex = 0;
		if(totalRows <= 1){
			return endDataIndex;
		}
		
		long firstDataIndex = getFirstRowIndexInPage();
		
		if(totalRows >= pageSize){
			// 带分页
			if(hasNextPage()){
				endDataIndex = firstDataIndex + pageSize - 1;
			} else {
				endDataIndex = firstDataIndex + (totalRows - (pageCount - 1)*pageSize) - 1;
			}
		} else {
			// 不带分页
			endDataIndex = totalRows-1;
		}
		return endDataIndex;
	}
	
	@Override
	public boolean hasNextPage(){
		return this.pageIndex < this.pageCount;
	}
	
	@Override
	public boolean hasPreviousPage(){
		return this.pageIndex > 1;
	}
	
	@Override
	public List<Object> getDatasObject(){
		throw new UnsupportedOperationException("error: no implements.");
	}
	
	public String toString(){
		return new StringBuilder().append("pager: {pageIndex=").append(pageIndex)
				.append(", pageSize=").append(pageSize)
				.append(", pageCount=").append(pageCount)
				.append(", totalRows=").append(totalRows)
				.append("}").toString();
	}
}
