package com.walker.db.page;

import com.walker.infrastructure.core.NamedThreadLocal;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;
import java.util.Map;

/**
 * 可对集合数据分页的上下文对象定义</p>
 * 它能提供对数据分页的支持，通常使用数据库分页，但也提供对其他方式的分页，如：文件数据。</br>
 * 这是对可分页内容的一种简单抽象，系统默认提供了对数据库的分页支持。
 * @author shikeying
 * @date 2013-9-24
 *
 */
public abstract class ListPageContext {
	
	private static final transient Logger logger = LoggerFactory.getLogger(ListPageContext.class);

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// 分页参数的参数名称定义，通常在UI中会搜集这些数据
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * 当前页-参数名称
	 */
	public static final String PARAM_NAME_PAGE_INDEX = "walker_page_index";
	
	/**
	 * 每页显示条数-参数名称
	 */
	public static final String PARAM_NAME_PAGE_SIZE = "walker_page_size";
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// 默认值定义
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	public static final int DEFAULT_PAGE_SIZE = 1;
	
	private static final NamedThreadLocal<Integer> pageIndexLocals 
		= new NamedThreadLocal<Integer>("pageIndexThreadLocal");
			
	private static final NamedThreadLocal<Integer> pageSizeLocals 
		= new NamedThreadLocal<Integer>("pageSizeThreadLocal");

	private static final NamedThreadLocal<PageSearch> pageSearchLocals
		= new NamedThreadLocal<PageSearch>("pageSearchThreadLocal");

	/**
	 * 从当前线程中返回当前页
	 * @return
	 */
	public static final int getCurrentPageIndex(){
		Integer val = pageIndexLocals.get();
		if(val == null)
			return Pager.DEFAULT_PAGE_INDEX;
		return val.intValue();
	}
	
	/**
	 * 从当前线程中返回[每页记录数]
	 * @return
	 */
	public static final int getCurrentPageSize(){
		Integer val = pageSizeLocals.get();
		if(val == null)
			return Pager.DEFAULT_PAGE_SIZE;
		return val.intValue();
	}
		
	/**
	 * 保存当前页信息到线程中
	 * @param pageIndex
	 */
	public static void setCurrentPageIndex(int pageIndex){
		Integer _old = pageIndexLocals.get();
		if(logger.isDebugEnabled() && _old != null){
			logger.debug("exist pageIndex in '" + pageIndexLocals + "' : " + _old);
		}
		pageIndexLocals.set(pageIndex);
	}
	
	/**
	 * 保存分页条数到线程中
	 * @param pageSize
	 */
	public static void setCurrentPageSize(int pageSize){
		Integer _old = pageSizeLocals.get();
		if(logger.isDebugEnabled() && _old != null){
			logger.debug("exist pageIndex in '" + pageSizeLocals + "' : " + _old);
		}
		pageSizeLocals.set(pageSize);
	}
	
	public static void clearCurrentPageIndex(){
		pageIndexLocals.remove();
	}
	
	public static void clearCurrentPageSize(){
		pageSizeLocals.remove();
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// 前端分页搜索条件相关，2022-11-16
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	public static final PageSearch getPageSearch(){
		return pageSearchLocals.get();
	}

	public static final void setPageSearch(PageSearch pageSearch){
		pageSearchLocals.set(pageSearch);
	}

	public static final void clearPageSearch(){
		pageSearchLocals.remove();
	}
		
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// 创建分页对象
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~	
	
	/**
	 * 返回带泛型的分页对象
	 * @param datas 业务数据，如：List<T> data
	 * @param pageIndex 页码值
	 * @param pageSize 每页显示记录数
	 * @param totalRows 总记录数
	 * @return
	 */
	public static final <T> GenericPager<T> createGenericPager(List<T> datas, int pageIndex, int pageSize, int totalRows){
//		int _pageInde = getContextPageIndex(pageIndex);
//		int _pageSize = getContextPageSize(pageSize);
//		int startRowIndex = (_pageInde - _pageInde % _pageSize)/_pageSize;
		return new GenericPager<T>(datas
				, getContextPageIndex(pageIndex)
				, getContextPageSize(pageSize), totalRows);
	}
	
	/**
	 * 返回带泛型的分页对象
	 * @param datas 业务数据，如：List<T> data
	 * @param pageIndex 页码值
	 * @param totalRows 总记录数
	 * @return
	 */
	public static final <T> GenericPager<T> createGenericPager(List<T> datas, int pageIndex, int totalRows){
		return new GenericPager<T>(datas
				, getContextPageIndex(pageIndex)
				, getContextPageSize(0), totalRows);
	}
	
	public static final <T> GenericPager<T> createGenericPager(int pageIndex, int pageSize, int totalRows){
		return new GenericPager<T>(null
				, getContextPageIndex(pageIndex)
				, getContextPageSize(pageSize), totalRows);
	}
	
	public static final <T> GenericPager<T> createEmptyGenericPager(){
		return new GenericPager<T>(null, 1, getContextPageSize(0), 0); 
	}
	
	/**
	 * 创建分页对象，业务类型为<code>Map&lt;String, Object&gt;</code>
	 * @param datas 业务数据，如：List&lt;Map&lt;String, Object&gt;&gt; data
	 * @param pageIndex 页码值
	 * @param pageSize 每页显示记录数
	 * @param totalRows 总记录数
	 * @return
	 */
	public static final MapPager createMapPager(List<Map<String, Object>> datas, int pageIndex, int pageSize, int totalRows){
		return new MapPager(datas
				, getContextPageIndex(pageIndex)
				, getContextPageSize(pageSize), totalRows);
	}
	
	/**
	 * 创建分页对象，业务类型为<code>Map&lt;String, Object&gt;</code>
	 * @param datas
	 * @param pageIndex
	 * @param totalRows
	 * @return
	 */
	public static final MapPager createMapPager(List<Map<String, Object>> datas, int pageIndex, int totalRows){
		return new MapPager(datas
				, getContextPageIndex(pageIndex)
				, getContextPageSize(0), totalRows);
	}
	
	/**
	 * 创建UI层可用的分页展示对象
	 * @param pager
	 * @param jsMethod
	 * @return
	 */
	public static final <T> PagerView<T> createPagerView(GenericPager<T> pager, String jsMethod){
		return new PagerView<T>(pager, jsMethod);
	}
	
	private static int getContextPageSize(int pageSizeValue){
		Integer _val = pageSizeLocals.get();
		if(_val != null)
			return _val.intValue();
		if(pageSizeValue <= 0)
			return Pager.DEFAULT_PAGE_SIZE;
		return pageSizeValue;
	}
	
	private static int getContextPageIndex(int pageIndexValue){
		Integer pageIndex = pageIndexLocals.get();
		if(pageIndex != null)
			return pageIndex.intValue();
		if(pageIndexValue <= 0)
			return Pager.DEFAULT_PAGE_INDEX;
		return pageIndexValue;
	}

}
