package com.walker.infrastructure.arguments.support;

import com.walker.infrastructure.arguments.ArgumentsException;
import com.walker.infrastructure.arguments.Variable;
import com.walker.infrastructure.arguments.VariableType;
import com.walker.infrastructure.utils.StringUtils;

public class DefaultVariable implements Variable {

	private String id;
	private String description = StringUtils.EMPTY_STRING;
	private VariableType type;
	
	private String stringValue;
	private int integerValue = 0;
	private boolean booleanValue = false;
	private float floatValue = 0F;
	private long longValue = 0;
	private double doubleValue = 0.0;
	
	private Object defaultValue = null;
	
	public DefaultVariable(){
		
	}
	
	public DefaultVariable(VariableType type, Object value, Object defaultValue){
		assert (type != null);
		assert (value != null);
		this.type = type;
		this.defaultValue = defaultValue;
		setValue(value);
	}
	
	public DefaultVariable setType(VariableType type) {
		this.type = type;
		return this;
	}

	public DefaultVariable setDefaultValue(Object defaultValue) {
		this.defaultValue = defaultValue;
		return this;
	}

	public DefaultVariable setValue(Object value){
		assert (type != null);
		if(type == VariableType.Integer){
			integerValue = Integer.parseInt(value.toString());
		} else if(type == VariableType.Boolean){
			booleanValue = Boolean.valueOf(value.toString());
		} else if(type == VariableType.Float){
			floatValue = Float.parseFloat(value.toString());
		} else if(type == VariableType.String){
			stringValue = value.toString();
		} else if(type == VariableType.Long){
			longValue = Long.parseLong(value.toString());
		} else if(type == VariableType.Double){
			doubleValue = Double.parseDouble(value.toString());
		} else
			throw new IllegalArgumentException("unsupported primitive type in value: " + value);
		// 默认情况：默认值与设置值一样
//		this.defaultValue = value;
		return this;
		
//		Class<?> dataClazz = value.getClass();
//		if(dataClazz.isPrimitive()){
//		} else {
//			if(dataClazz == String.class){
//				stringValue = value.toString();
//			} else if(dataClazz == Integer.class){
//				integerValue = (Integer)value;
//			} else if(dataClazz == Boolean.class){
//				booleanValue = (Boolean)value;
//			} else if(dataClazz == Float.class){
//				floatValue = (Float)value;
//			} else
//				throw new IllegalArgumentException("unsupported primitive type: " + dataClazz);
//		}
	}
	
	public DefaultVariable setId(String id) {
		assert (StringUtils.isNotEmpty(id));
		this.id = id;
		return this;
	}

	public DefaultVariable setDescription(String description) {
		this.description = description;
		return this;
	}

	@Override
	public String getId() {
		return id;
	}

	@Override
	public String getDescription() {
		return this.description;
	}

	@Override
	public VariableType getType() {
		return type;
	}

	@Override
	public String getStringValue() {
		if(type != VariableType.String){
			throw new ArgumentsException("error invoke in getStringValue(), type: " + type);
		}
		return this.stringValue;
	}

	@Override
	public int getIntegerValue() {
		if(type != VariableType.Integer){
			throw new ArgumentsException("error invoke in getStringValue(), type: " + type);
		}
		return this.integerValue;
	}

	@Override
	public boolean getBooleanValue() {
		if(type != VariableType.Boolean){
			throw new ArgumentsException("error invoke in getStringValue(), type: " + type);
		}
		return this.booleanValue;
	}

	@Override
	public float getFloatValue() {
		if(type != VariableType.Float){
			throw new ArgumentsException("error invoke in getStringValue(), type: " + type);
		}
		return this.floatValue;
	}

	@Override
	public Object getDefaultValue() {
		if(defaultValue == null)
			return null;
		if(type == VariableType.String){
			return this.defaultValue.toString();
		} else if(type == VariableType.Integer){
			return Integer.parseInt(defaultValue.toString());
		} else if(type == VariableType.Boolean){
			return Boolean.valueOf(defaultValue.toString());
		} else if(type == VariableType.Float){
			return Float.parseFloat(defaultValue.toString());
		} else if(type == VariableType.Long){
			return Long.parseLong(defaultValue.toString());
		} else
			throw new IllegalArgumentException("unsupported defaultValue type: " + defaultValue.getClass());
	}

	public String toString(){
		return new StringBuilder().append("{")
				.append("id = ").append(id)
				.append(", desc = ").append(description)
				.append(", type = ").append(type)
				.append("}").toString();
	}
	
	public boolean equals(Object o){
		if(o == null) return false;
		if(o == this) return true;
		if(o instanceof DefaultVariable){
			DefaultVariable dv = (DefaultVariable)o;
			if(dv.id.equals(this.id) && dv.description.equals(this.description) 
					&& dv.type == this.type){
				return true;
			}
		}
		return false;
	}
	
	public int hashCode(){
		return 31 + 31*this.id.hashCode() + this.description.hashCode() + type.hashCode();
	}

	@Override
	public long getLongValue() {
		return this.longValue;
	}

	@Override
	public double getDoubleValue() {
		return this.doubleValue;
	}
}
