package com.walker.infrastructure.arguments.support;

import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import com.walker.infrastructure.core.DomParser;
import com.walker.infrastructure.core.domx.Dom4jParser;
//import com.walker.infrastructure.core.domx.Jdom2Parser;

import com.walker.infrastructure.arguments.ArgumentsException;
import com.walker.infrastructure.arguments.Group;
import com.walker.infrastructure.arguments.Variable;
import com.walker.infrastructure.arguments.VariableType;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.infrastructure.utils.WorkingTimeTester;
import org.dom4j.Attribute;
import org.dom4j.Document;
import org.dom4j.Element;
import org.dom4j.Node;
import org.dom4j.io.OutputFormat;
import org.dom4j.io.XMLWriter;

public class XmlArgumentsManager extends FileArgumentsManager {

	private final DomParser<Document> domParser = new Dom4jParser();
//	private final DomParser<Document> domParser = new Jdom2Parser();

	private Document document;
	
	private final OutputFormat format = OutputFormat.createPrettyPrint();
	
	@SuppressWarnings("unchecked")
	@Override
	protected List<Group> loadFile(InputStream inputStream) throws Exception {
		document = domParser.getDocumentFromStream(inputStream);
		Element e = document.getRootElement();
		if(e == null) return null;
		
		List<Group> groupList = new ArrayList<Group>(8);
		for(Iterator<Element> j = e.elementIterator(); j.hasNext();){
			groupList.add(loadGroups(j.next()));
		}
		return groupList;
	}
	
	@SuppressWarnings("unchecked")
	private Group loadGroups(Element groupElement){
		String name = groupElement.getName().toLowerCase();
		if(!name.equalsIgnoreCase(GROUP_NAME))
			throw new IllegalArgumentException("unsupported group name: " + name);
		
		DefaultGroup _group = new DefaultGroup();
		
		Attribute attr = null;
		for(Iterator<Attribute> a = groupElement.attributeIterator(); a.hasNext();){
			attr = a.next();
			if(attr.getName().equalsIgnoreCase(GROUP_ID_ATTR)){
				_group.setId(attr.getValue().trim());
			} else if(attr.getName().equalsIgnoreCase(GROUP_NAME_ATTR)){
				_group.setName(attr.getValue().trim());
			} else if(attr.getName().equalsIgnoreCase(GROUP_ORDER_ATTR)){
				_group.setOrder(Integer.parseInt(attr.getValue().trim()));
			}
		}
		
		/* 加载分组下的参数子节点 */
		List<Variable> list = loadVariables(groupElement);
		_group.setChildren(list);
		logger.info("--------------------- 【开始】加载参数分组：" + name);
		for(Variable v : list){
			logger.info(v.toString());
		}
//		logger.info("--------------------- 【结束】加载参数分组：" + name);
		return _group;
	}
	
	@SuppressWarnings("unchecked")
	private List<Variable> loadVariables(Element groupElement){
		if(groupElement.elements() != null){
			Element varElement = null;
			List<Variable> varList = new ArrayList<Variable>(groupElement.elements().size());
			
			for(Iterator<Element> j = groupElement.elementIterator(); j.hasNext();){
				varElement = j.next();
				
				if(varElement.getName().equalsIgnoreCase(VAR_NAME)){
					Attribute attr = null;
					DefaultVariable variable = new DefaultVariable();
					for(Iterator<Attribute> a = varElement.attributeIterator(); a.hasNext();){
						attr = a.next();
						if(attr.getName().equalsIgnoreCase(VAR_ID_ATTR)){
							variable.setId(attr.getValue().trim());
						} else if(attr.getName().equalsIgnoreCase(VAR_NAME_ATTR)){
							variable.setDescription(attr.getValue().trim());
						} else if(attr.getName().equalsIgnoreCase(VAR_TYPE_ATTR)){
							variable.setType(VariableType.getType(attr.getValue().trim()));
						} else if(attr.getName().equalsIgnoreCase(VAR_VALUE_ATTR)){
							variable.setValue(attr.getValue().trim());
						} else if(attr.getName().equalsIgnoreCase(VAR_DEFAULT_ATTR)){
							variable.setDefaultValue(attr.getValue().trim());
						} else
							throw new IllegalArgumentException("unsupported variable attribute: " + attr.getName());
					}
					varList.add(variable);
					
				} else
					throw new IllegalArgumentException("unsupported variable name: " + varElement.getName());
			}
			return varList;
		}
		return null;
	}

	@Override
	protected void saveVariable(String groupId, String variableId, Object value)
			throws Exception {
		updateDocument(groupId, variableId, value);
		writeFile();
	}

	@Override
	protected void saveVariables(List<Object[]> changedList) throws Exception {
		for(Object[] args : changedList){
			updateDocument(args[0].toString(), args[1].toString(), args[2]);
		}
		writeFile();
	}
	
	@Override
	protected void insertVariables(List<Object[]> insertList) throws Exception {
		Element root = null;
		Group group = null;
		Variable variable = null;
		for(Object[] objects : insertList){
			group = (Group)objects[0];
			variable = (Variable)objects[1];
			Node node = document.selectSingleNode(getGroupNodeString(group.getId()));
			if(node == null){
				logger.debug("group不存在，需要创建: " + group);
				root = document.getRootElement();
				Element groupElement = root.addElement(GROUP_NAME);
				groupElement.addAttribute(GROUP_ID_ATTR, group.getId());
				groupElement.addAttribute(GROUP_NAME_ATTR, group.getName());
				groupElement.addAttribute(GROUP_ORDER_ATTR, String.valueOf(group.getOrder()));
				node = groupElement;
			}
			// 处理变量
			Node varNode = document.selectSingleNode(getVariableNodeString(group.getId(), variable.getId()));
			if(varNode != null){
				logger.warn("variable is exist, it can't be added: " + variable);
				continue;
			}
			Element varElement = ((Element)node).addElement(VAR_NAME);
			varElement.addAttribute(VAR_ID_ATTR, variable.getId());
			varElement.addAttribute(VAR_NAME_ATTR, variable.getDescription());
			varElement.addAttribute(VAR_TYPE_ATTR, variable.getType().getTypeName());
			if(variable.getType() == VariableType.String){
				varElement.addAttribute(VAR_VALUE_ATTR, variable.getStringValue());
			} else if(variable.getType() == VariableType.Boolean){
				varElement.addAttribute(VAR_VALUE_ATTR, String.valueOf(variable.getBooleanValue()));
			} else if(variable.getType() == VariableType.Integer){
				varElement.addAttribute(VAR_VALUE_ATTR, String.valueOf(variable.getIntegerValue()));
			} else if(variable.getType() == VariableType.Float){
				varElement.addAttribute(VAR_VALUE_ATTR, String.valueOf(variable.getFloatValue()));
			} else if(variable.getType() == VariableType.Long){
				varElement.addAttribute(VAR_VALUE_ATTR, String.valueOf(variable.getLongValue()));
			}
			varElement.addAttribute(VAR_DEFAULT_ATTR, variable.getDefaultValue().toString());
		}
		writeFile();
	}
	
	private String getGroupNodeString(String groupId){
		StringBuilder path = new StringBuilder("root/");
		path.append("group[@id='");
		path.append(groupId);
		path.append("']");
		return path.toString();
	}
	
	private String getVariableNodeString(String groupId, String variableId){
		StringBuilder path = new StringBuilder("root/");
		path.append("group[@id='");
		path.append(groupId);
		path.append("']/variable[@id='");
		path.append(variableId);
		path.append("']");
		return path.toString();
	}

	private void updateDocument(String groupId, String variableId, Object value){
		assert (StringUtils.isNotEmpty(groupId));
		assert (StringUtils.isNotEmpty(variableId));
		assert (value != null);
		
		Node node = document.selectSingleNode(getVariableNodeString(groupId, variableId));
		if(node == null){
			throw new RuntimeException("Node not found: " + groupId + " --> " + variableId);
		}
		logger.debug("node = " + node.asXML());
		Element element = (Element)node;
		element.addAttribute(VAR_VALUE_ATTR, value.toString());
	}
	
	private void writeFile(){
		XMLWriter output = null;
		format.setEncoding("UTF-8");
		try { 
			//用stream字节流的形式，让编码格式设置生效
			output = new XMLWriter(new FileOutputStream(getFilePath()), format);
			output.write(document);
		} catch (IOException e) {
			e.printStackTrace();
			throw new ArgumentsException("写入文件出现错误: " + e.getMessage(), e);
		} finally {
			if(output != null){
				try {
					output.close();
				} catch (IOException e) {
				}
			}
		}
	}

	private static final String GROUP_NAME = "group";
	private static final String GROUP_ID_ATTR = "id";
	private static final String GROUP_NAME_ATTR = "name";
	private static final String GROUP_ORDER_ATTR = "order";
	
	private static final String VAR_NAME = "variable";
	private static final String VAR_ID_ATTR = "id";
	private static final String VAR_NAME_ATTR = "name";
	private static final String VAR_TYPE_ATTR = "type";
	private static final String VAR_VALUE_ATTR = "value";
	private static final String VAR_DEFAULT_ATTR = "default";
	
	public String toString(){
		return this.getFilePath();
	}
	
	/**
	 * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	 */
	public static void main(String[] args) throws Exception{
		XmlArgumentsManager argumentManager = new XmlArgumentsManager();
		argumentManager.setClasspathFileName("variable.xml");
		argumentManager.setSource("test");
		argumentManager.afterPropertiesSet();
		System.out.println(argumentManager.getVariableList("SYSTEM.SUPERPASS"));
		
		System.out.println("+++ 分组信息: " + argumentManager.getGroupList());
		
		WorkingTimeTester tester = new WorkingTimeTester("var");
		Variable var = argumentManager.getVariable("11");
		System.out.println("--分页数：" + var.getIntegerValue());
		tester.stop();
		
		Object[] args1 = new Object[]{"1", "11", 88};
		Object[] args2 = new Object[]{"2", "21", "修改过企业名称"};
		List<Object[]> argList = new ArrayList<Object[]>();
		argList.add(args1);
		argList.add(args2);
		argumentManager.persist(argList);
//		argumentManager.persist("1", "11", 56);
		Variable v2 = argumentManager.getVariable("21");
		System.out.println("修改后缓存: " + v2.getStringValue());
		
		// 添加一个节点
		DefaultGroup group2 = new DefaultGroup();
		group2.setId("testGroup2");
		group2.setName("系统提醒");
		group2.setOrder(8);
		
		DefaultVariable v = new DefaultVariable();
		v.setId("smtp.mail.address");
		v.setDescription("邮件发送地址");
		v.setType(VariableType.String);
		v.setValue("smtp.163.com");
		v.setDefaultValue("smtp.mymail.net");
		
		List<Object[]> insertList = new ArrayList<Object[]>();
		insertList.add(new Object[]{group2, v});
		argumentManager.insert(insertList);
	}

}
