package com.walker.push;

import com.walker.infrastructure.utils.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;

public abstract class AbstractPushObject<T> implements Pushable<T>{

    protected transient final Logger logger = LoggerFactory.getLogger(getClass());

    @Override
    public String getId() {
        return this.id;
    }

    @Override
    public String getName() {
        return this.name;
    }

    @Override
    public PushResult push(Notification notification) throws PushException {
        if(notification == null){
            throw new IllegalArgumentException("推送消息未提供");
        }
        if(StringUtils.isEmpty(notification.getContent())
                || StringUtils.isEmptyList(notification.getChannelList())){
            throw new IllegalArgumentException("推送消息缺少必要参数");
        }
        if(!notification.getBroadcast() && StringUtils.isEmptyList(notification.getReceiverList())){
            throw new IllegalArgumentException("非广播推送必须指定接收者");
        }

        boolean hasChannel = false;
        for(NotificationChannel channel : notification.getChannelList()){
            if(channel == this.getNotificationChannel()){
                hasChannel = true;
                break;
            }
        }
        if(!hasChannel){
            throw new PushException(notification.getId(), "当前消息不支持该推送者：" + this.name, null);
        }

        // 转换成具体发送的业务特定对象
        List<T> list = this.translateToTarget(notification);
        if(StringUtils.isEmptyList(list)){
//            throw new PushException(notification.getId(), "转换业务通知失败: list = null", null);
            logger.warn("translateToTarget():转换返回空对象");
        }

        // 2023-04-24 异步操作处理
        if(this.supportAsync){
            if(this.pushStatusListener == null){
                throw new IllegalStateException("异步推送，需要先添加：pushStatusListener");
            }
        }

        // 调用发送请求
        PushResult pushResult = this.doPushContent(notification, list);
//        if(!StringUtils.isEmptyList(pushResult.getFailedList())){
//            this.logger.error("推送消息失败，id={}, failed={}", notification.getId(), pushResult.getFailedList());
//            if(!notification.getBroadcast()){
//                // 非广播消息，需要检查是否全部失败，成功的需要保存
//                List<String> successList = new ArrayList<>(8);
//                for(String one : notification.getReceiverList()){
//                    if(!pushResult.getFailedList().contains(one)){
//                        successList.add(one);
//                    }
//                }
//                this.persistent(notification, successList);
//                logger.debug("保存了（部分）成功推送集合");
//            }
//        } else {
//            // 全部成功，保存
//            this.persistent(notification, notification.getReceiverList());
//            logger.debug("保存了（全部）成功推送集合");
//        }
        return pushResult;
    }

    /**
     * 执行具体推送的方法，由子类完成。
     * @param notification
     * @param data 转换成业务通知集合
     * @return 只有同步类型时，返回的才是正确调用结果，异步需要看监听器
     * @throws PushException
     * @date 2023-04-22
     */
    protected abstract PushResult doPushContent(Notification notification, List<T> data) throws PushException;

//    /**
//     * 持久化保存推送数据（到数据库）
//     * @param notification
//     * @param successReceiverList
//     */
//    protected abstract void persistent(Notification notification, List<String> successReceiverList);

//    protected abstract void onException(Notification notification, String error);
//    protected abstract void onSuccess(Notification notification, Object option);

    @Override
    public int hashCode(){
        return this.id.hashCode();
    }

    @Override
    public boolean equals(Object obj){
        if(obj == null){return false;}
        if(obj instanceof Pushable){
            Pushable push = (Pushable) obj;
            if(push.getId().equals(this.id)){
                return true;
            }
        }
        return false;
    }

    public void setId(String id) {
        this.id = id;
    }

    public void setName(String name) {
        this.name = name;
    }

    @Override
    public boolean supportAsync() {
        return supportAsync;
    }

    public void setSupportAsync(boolean supportAsync) {
        this.supportAsync = supportAsync;
    }

    @Override
    public PushStatusListener getPushStatusListener() {
        return pushStatusListener;
    }

    @Override
    public void setPushStatusListener(PushStatusListener pushStatusListener) {
        this.pushStatusListener = pushStatusListener;
    }

    private PushStatusListener pushStatusListener;
    private boolean supportAsync = false;
    private String id;
    private String name;
}
