package com.walker.di;

import java.util.List;

/**
 * 数据导入器定义。<p></p>
 * 可用于导入外部资源到目标位置，如:
 * <pre>
 *     1)从csv导入到数据库表
 *     2)从数据库到数据库表
 *     3)其他等
 * </pre>
 * 该对象为有状态，需要在每次使用时创建实例，并调用实例的load方法。
 * @author 时克英
 * @date 2023-01-31
 */
public interface DataImportor {

    /**
     * 加载并导入数据，业务调用该方法完成导入具体操作。
     * @throws DataImportException
     */
    void load() throws DataImportException;

    /**
     * 是否支持批量导入，对于大数据量导入会提高效率。<p></p>
     * 此时导入策略会启用批量加载并小块写入方式，避免内存溢出并提高效率。
     * <p></p>
     *
     * @return
     */
    boolean isBatchSupport();

    /**
     * 是否支持断点续传(导入)，如果支持，引擎会记录导入位置，并在下次重新导入时，从该位置开始。<p></p>
     * 但这需要提供唯一标识以确定要使用断点功能，
     * @return
     */
    boolean isBrokenContinue();

    /**
     * 设置允许批量导入，使用默认参数。
     */
    void setBatchEnabled();

    /**
     * 设置批量导入的数量，默认不能超过:512
     * @param batchSize
     */
    void setBatchSize(long batchSize);

    /**
     * 设置是否支持断点续传(导入)
     * @param allow
     */
    void setBrokenContinue(boolean allow);

    /**
     * 设置断点续传的标识，用于告知引擎需要使用该功能。
     * @param brokenId
     */
    void setBrokenId(String brokenId);

    /**
     * 获取导入数据的列名字集合。
     * @return
     */
    List<String> getFieldNames();

    /**
     * 设置导入数据列名字
     * @param fieldNames
     */
    void setFieldNames(List<String> fieldNames);

    /**
     * 返回批量加载的数量。
     * @return
     */
    long getBatchSize();

    /**
     * 返回批量加载错误，因为会多次持续执行，因此有可能整个过程执行完才能获得结果。需要测试
     * @return
     * @date 2023-02-01
     */
    String getBatchError();

    /**
     * 设置导入数据时，忽略的行数。<p></p>
     * 有时导入文件第一个可能是说明行，需要忽略，该方法可设置忽略多少行(这些行不导入)。
     * @param number
     */
    void setIgnoreRows(int number);

    /**
     * 设置批量导入时，每次保存间隔线程休眠时间，如果为0表示不休息。
     * @param millSeconds 毫秒值
     */
    void setBatchSleepMills(long millSeconds);

    /**
     * 返回已经成功导入数量。
     * @return
     */
    long getSuccessSize();

    /**
     * 导入唯一标识，由业务确定，在导入错误时，运行把错误结果重新导入
     * @return
     */
    String getId();

    void setId(String id);

    /**
     * 返回导入错误结果文件路径，如: d:/file/12344566_error.csv
     * @return
     */
    String getErrorFile();

    /**
     * 设置'是否显示错误结果文件'，如果显示则会为用户提供导入错误的结果文件。
     * @param showError
     * @date 2023-02-02
     */
    void setShowError(boolean showError);

    /**
     * 设置保存文件的文件夹路径，如: d:/file/
     * @param saveFileFolder
     */
    void setSaveFileFolder(String saveFileFolder);

    /**
     * 设置导入数据中，表头有几行，支持多行表头。
     * @param headRowNumber
     * @date 2023-02-03
     */
    void setHeadRowNumber(int headRowNumber);

    /**
     * 设置数据更新方式，如果在导入时需要判断数据是否存在，则可以设置该更新策略。<p></p>
     * 目前支持两种: 忽略、覆盖。
     * @param updateType
     * @date 2023-02-05
     */
    void setUpdateType(UpdateType updateType);
}
