package com.walker.infrastructure.arguments;

import java.util.List;

import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;

//import com.walker.beans.factory.DisposableBean;
//import com.walker.beans.factory.InitializingBean;

/**
 * 参数管理器定义</p>
 * 该管理器实现了如何加载、获取、持久化保存可变参数的功能。</br>
 * 子类实现具体的加载数据过程，如：xml文件、数据库等。
 * @author shikeying
 * @date 2013-11-28
 *
 */
public interface ArgumentsManager extends InitializingBean, DisposableBean {

	/**
	 * 根据可变参数ID，返回该参数对象
	 * @param id
	 * @return
	 * 
	 * @throws ElementNotFoundException
	 */
	Variable getVariable(String id);
	
	/**
	 * 持久化一个变量值，也就是更新一个现有的变量。</br>
	 * 此方法适用于配置中仅有一个分组，默认就持久化到第一个分组中。
	 * @param variableId
	 * @param value
	 */
	void persist(String variableId, Object value);
	
	/**
	 * 持久化一个变量值，也就是更新一个现有的变量。
	 * @param groupId 分组ID，必须存在
	 * @param variableId 变量ID
	 * @param value 新的变量值
	 * 
	 * @throws ArgumentsException 如果保存失败，会抛出此运行时异常。
	 */
	void persist(String groupId, String variableId, Object value);
	
	/**
	 * 持久化批量参数更新，集合中是数组对象，Object[]{groupId, variableId, value}</br>
	 * @param changedList
	 * @throws ArgumentsException 如果保存失败，会抛出此运行时异常。
	 */
	void persist(List<Object[]> changedList);
	
	/**
	 * 加入新的参数集合，集合中是数组对象，Object[]{group, variable}</br>
	 * 如果已经存在该参数，则不再创建。
	 * @param insertList
	 * @throws ArgumentsException 如果保存失败，会抛出此运行时异常。
	 */
	void insert(List<Object[]> insertList);

	/**
	 * 返回可变参数的所有分组信息，如果不存在返回<code>null</code>
	 * @return
	 */
	List<Group> getGroupList();
	
	/**
	 * 返回某个分组下所有可变参数集合，如果不存在返回<code>null</code>
	 * @param groupId
	 * @return
	 */
	List<Variable> getVariableList(String groupId);
	
	/**
	 * 设置可变参数加载的原始信息，如：文件名、数据源地址等。</p>
	 * 系统会实现至少两种加载源：
	 * <li>从XML配置文件中加载参数</li>
	 * <li>从数据库中加载参数</li>
	 * @param source
	 */
	void setSource(Object source);
}
