package com.walker.infrastructure.utils;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.util.Calendar;
import java.util.Date;

/**
 * 处理日期格式以及转换的工具类
 * @author shikeying
 * @date 2013-11-21
 *
 */
public abstract class DateUtils {

	public static final String TEXT_DATE_WITH_LINE = "yyyy-MM-dd";
	public static final String TEXT_DATE = "yyyyMMdd";
	public static final String TEXT_DATE_TIME_WITH_LINE = "yyyy-MM-dd HH:mm:ss";
	public static final String TEXT_DATE_TIME = "yyyyMMddHHmmss";
	public static final String TEXT_DATE_START = "yyyyMMdd000000";
	public static final String TEXT_MONTH_START = "yyyyMM01000000";
	public static final String TEXT_MONTH_END = "yyyyMMdd235959";
	public static final String TEXT_YEAR_START = "yyyy0101000000";
	public static final String TEXT_YEAR_END = "yyyy1231235959";
//	private static final DateFormat whippletreeDateFormat = new SimpleDateFormat("yyyy-MM-dd");
//	private static final DateFormat whippletreeTimeFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
//	private static final DateFormat whippleTimeFormat = new SimpleDateFormat("yyyyMMddHHmmss");
//	private static final DateFormat whippleDateFormat = new SimpleDateFormat("yyyyMMdd");
//	private static final DateTimeFormatter whippletreeTimeFormat = DateTimeFormatter.ofPattern("yyyy-MMM-dd HH:mm:ss");

	private static final String LATE_OF_DAY = " 23:59:59";
	public static final String DATE_SEPARATOR_OLD = "/";
	public static final String DATE_SEPARATOR = "-";

	/**
	 * 返回上年份第一天时间值
	 * @return 如：20230101000000
	 * @date 2023-06-02
	 */
	public static long getLastYearStartDay() {
		Calendar c = Calendar.getInstance();
		c.add(Calendar.YEAR, -1);
		SimpleDateFormat startSdf = new SimpleDateFormat(TEXT_YEAR_START);
		return Long.parseLong(startSdf.format(c.getTime()));
	}
	public static final long getLastYearEndDay() {
		Calendar c = Calendar.getInstance();
		c.add(Calendar.YEAR, -1);
		SimpleDateFormat endSdf = new SimpleDateFormat(TEXT_YEAR_END);
		return Long.parseLong(endSdf.format(c.getTime()));
	}

	/**
	 * 返回当前年份第一天时间值
	 * @return 如：20230101000000
	 * @date 2023-06-02
	 */
	public static final long getCurrentYearStartDay(){
		SimpleDateFormat dft = new SimpleDateFormat(TEXT_YEAR_START);
		return Long.parseLong(dft.format(new Date()));
	}
	public static final long getCurrentYearEndDay(){
		SimpleDateFormat dft = new SimpleDateFormat(TEXT_YEAR_END);
		return Long.parseLong(dft.format(new Date()));
	}

	/**
	 * 返回上月第一天的时间值
	 * @return 如：20230601000000
	 * @date 2023-06-02
	 */
	public static final long getLastMonthStartDay() {
		Calendar c = Calendar.getInstance();
		c.add(Calendar.MONTH, -1);
		c.set(Calendar.DAY_OF_MONTH, 1);
		SimpleDateFormat startSdf = new SimpleDateFormat(TEXT_MONTH_START);
		return Long.parseLong(startSdf.format(c.getTime()));
	}
	public static final long getLastMonthEndDay() {
		Calendar c = Calendar.getInstance();
		c.add(Calendar.MONTH, -1);
		c.set(Calendar.DAY_OF_MONTH, c.getActualMaximum(Calendar.DAY_OF_MONTH));
		SimpleDateFormat endSdf = new SimpleDateFormat(TEXT_MONTH_END);
		return Long.parseLong(endSdf.format(c.getTime()));
	}

	/**
	 * 返回当前月份第一天的时间值
	 * @return 如：20230601000000
	 * @date 2023-06-02
	 */
	public static final long getCurrentMonthStartDay(){
		SimpleDateFormat dft = new SimpleDateFormat(TEXT_MONTH_START);
		return Long.parseLong(dft.format(new Date()));
	}
	public static final long getCurrentMonthEndDay() {
		Calendar c = Calendar.getInstance();
		c.set(Calendar.DAY_OF_MONTH, c.getActualMaximum(Calendar.DAY_OF_MONTH));
		SimpleDateFormat endSdf = new SimpleDateFormat(TEXT_MONTH_END);
		return Long.parseLong(endSdf.format(c.getTime()));
	}


	/**
	 * 获得本周第一天，格式：20230602000000
	 * @date 2023-06-02
	 */
	public static long getCurrentWeekStartDay() {
		Calendar c = Calendar.getInstance();
		c.add(Calendar.WEEK_OF_MONTH, 0);
		c.set(Calendar.DAY_OF_WEEK, 2);
		SimpleDateFormat startSdf = new SimpleDateFormat(TEXT_DATE_START);
		return Long.parseLong(startSdf.format(c.getTime()));
	}

	/**
	 * 把数值可视化时间转换成毫秒值，如：20140829102501 --> 毫秒值
	 * @param showTimeLong
	 * @return
	 */
	public static final long toMillSeconds(long showTimeLong){
		String showTimeStr = toShowDate(showTimeLong);
		return getDateLong(showTimeStr);
	}

	/**
	 * 把8位整形数值，转换成日期显示格式。</p>
	 * 如：20140829 --> 2014-08-29
	 * @param date
	 * @return
	 */
	public static final String toShowDate(int date){
		return LongCalendar.toString(date, DATE_SEPARATOR);
	}

	/**
	 * 把14位整形数值，转换成日期显示格式。</p>
	 * 如：20140829102501 --> 2014-08-29 10:25:01
	 * @param date
	 * @return
	 */
	public static final String toShowDate(long date){
		return LongCalendar.toString(date, DATE_SEPARATOR);
	}

	/**
	 * 把显示的日期格式转化成整型值。</p>
	 * 如：2014-08-29 --> 20140829
	 * @param showDate
	 * @return
	 */
	public static final int toIntegerDate(String showDate){
		return (int)LongCalendar.longCalender(showDate, DATE_SEPARATOR);
	}
	public static final int toIntegerDateTime(String showDate){
		return (int)LongCalendar.longCalender(showDate, DATE_SEPARATOR);
	}

	/**
	 * 把显示的日期格式转化成整型值。</p>
	 * 如：2014-08-29 --> 20140829000000
	 * @param showDate
	 * @return
	 */
	public static final long toLongDateTime(String showDate){
		return (LongCalendar.longCalender(showDate, DATE_SEPARATOR) * 1000000);
	}

	/**
	 * 返回当月第一天最早的时间毫秒数，如：2014-03-01 00:00:00 毫秒数
	 * @return
	 */
	public static final long getCurrentMonthFirstDayEarly(){
		return getDateLongEarly(LongDateHelper.getFirstDayOfMonth());
	}

	/**
	 * 返回当天最早的毫秒数，即：当天00:00:00的毫秒值
	 * @return
	 */
	public static final long getTodayLongEarly(){
		return getDateLongEarly(getTodayForHuman());
	}

	/**
	 * 返回当天最晚的毫秒数，即：当天23:59:59的毫秒值
	 * @return
	 */
	public static final long getTodayLongLate(){
		return getDateLongLate(getTodayForHuman());
	}

	/**
	 * 返回当天日期，如：2013-11-21
	 * @return
	 */
	public static final String getTodayForHuman(){
		return LongCalendar.getCurrentDateView().replaceAll(DATE_SEPARATOR_OLD, DATE_SEPARATOR);
	}

	/**
	 * 把毫秒值格式化为可以识别的日期字符串，如：2013-11-21 12:33:09
	 * @param millis
	 * @return
	 */
	public static final String getDateTimeForHuman(long millis){
		if(millis == 0) return "N/A";
//		LocalDateTime localDateTime = new Date(millis).toInstant().atOffset(ZoneOffset.of("+8")).toLocalDateTime();
		DateFormat whippletreeTimeFormat = new SimpleDateFormat(TEXT_DATE_TIME_WITH_LINE);
		return whippletreeTimeFormat.format(new Date(millis));
	}

	/**
	 * 把毫秒值格式化为可以识别的日期字符串，如：2013-11-21
	 * @param millis
	 * @return
	 */
	public static final String getDateForHuman(long millis){
		if(millis == 0) return "N/A";
		DateFormat whippletreeDateFormat = new SimpleDateFormat(TEXT_DATE_WITH_LINE);
		return whippletreeDateFormat.format(new Date(millis));
	}

	/**
	 * 返回给定毫秒数对应的日期数值，如：20131121
	 * @param millis
	 * @return
	 */
	public static final long getDateNumber(long millis){
		if(millis == 0) return 0;
		DateFormat whippleDateFormat = new SimpleDateFormat(TEXT_DATE);
		return Long.parseLong(whippleDateFormat.format(new Date(millis)));
	}

	/**
	 * 返回给定毫秒数对应的日期数值，如：20201220120130
	 * @param millis
	 * @return
	 * @date 2021-10-25
	 */
	public static final long getDateTimeNumber(long millis){
		if(millis == 0) return 0;
		DateFormat whippleTimeFormat = new SimpleDateFormat(TEXT_DATE_TIME);
		return Long.parseLong(whippleTimeFormat.format(new Date(millis)));
	}

	/**
	 * 返回当前日期数值，如：20201220120130
	 * @return
	 */
	public static final long getDateTimeNumber(){
		return getDateTimeNumber(System.currentTimeMillis());
	}

	/**
	 * 返回给定毫秒数对应的日期数值，如：20201220120130
	 * @param showDateTime 如：2020-12-20 12:01:30
	 * @return
	 */
	public static final long getDateTimeNumber(String showDateTime){
		long millis = getDateLong(showDateTime);
		DateFormat whippleTimeFormat = new SimpleDateFormat(TEXT_DATE_TIME);
		return Long.parseLong(whippleTimeFormat.format(new Date(millis)));
	}

	/**
	 * 返回给定日期，最晚的毫秒值。</p>
	 * 如：2013-11-21，系统会返回 2013-11-21 23:59:59 的毫秒值
	 * @param showDate 2013-11-21
	 * @return
	 */
	public static final long getDateLongLate(String showDate){
		String input = checkDate(showDate);
		input += LATE_OF_DAY;
		return getDateLong(input);
	}

	/**
	 * 返回给定日期，最早的毫秒值。</p>
	 * 如：2013-11-21，系统会返回 2013-11-21 00:00:00 的毫秒值
	 * @param showDate 2013-11-21
	 * @return
	 */
	public static final long getDateLongEarly(String showDate){
		return getDateLong(checkDate(showDate));
	}

	private static final String checkDate(String showDate){
		assert (StringUtils.isNotEmpty(showDate));
		String input = showDate.trim();
		if(input.indexOf(":") >= 0)
			throw new IllegalArgumentException("wrong input, format e.g. 2013-11-21.");
		return input;
	}

	/**
	 * 返回字符串日期的毫秒时间值。
	 * @param showDateTime 可以是：2013-11-21 or 2013-11-21 11:21:15
	 * @return
	 */
	public static final long getDateLong(String showDateTime){
		return getDate(showDateTime).getTime();
	}

	/**
	 * 把字符串日期格式化成<code>java.util.Date</code>对象。</p>
	 * 支持日期和时间两种格式，必须是以下形式:
	 * <pre>
	 * <li>2013-11-21</li>
	 * <li>2013-11-21 11:21:15</li>
	 * </pre>
	 * @param showDate 输入的日期字符串，必须用'-'作为分隔符。如：2013-11-21
	 * @return
	 */
	public static final Date getDate(String showDate){
		String input = showDate.trim();
		if(input.indexOf(DATE_SEPARATOR) == -1)
			throw new IllegalArgumentException("date of input must be separated width '-'");

		try {
			/* 判断是时间还是只有日期 */
			if(input.indexOf(":") > 0){
				DateFormat whippletreeTimeFormat = new SimpleDateFormat(TEXT_DATE_TIME_WITH_LINE);
				return whippletreeTimeFormat.parse(input);
			} else {
				DateFormat whippletreeDateFormat = new SimpleDateFormat(TEXT_DATE_WITH_LINE);
				return whippletreeDateFormat.parse(input);
			}
		} catch (ParseException e) {
			throw new RuntimeException("translate date error: " + showDate, e);
		}
	}

	/**
	 * 输入日期（或时间）字符串，返回年月字符串。</p>
	 * 如：2016-07-30 or 2016-07-30 10:01:36 ---> 201607
	 * @param showDateOrTime
	 * @return
	 */
	public static String getYearMonthValue(String showDateOrTime){
		return showDateOrTime.substring(0, 7).replace(DATE_SEPARATOR, StringUtils.EMPTY_STRING);
	}

	/**
	 * 返回当前年月字符串，如：201607
	 * @return
	 */
	public static String getYearMonthCurrentValue(){
		return getYearMonthValue(getDateForHuman(System.currentTimeMillis()));
	}

	/**
	 * 返回显示时间字符串，如：20161202152028
	 * @param millis 输入毫秒值
	 * @return
	 */
	public static String getDateTimeSecondForShow(long millis){
		if(millis == 0) return "N/A";
		DateFormat whippleTimeFormat = new SimpleDateFormat(TEXT_DATE_TIME);
		return whippleTimeFormat.format(new Date(millis));
	}

	public static String getDateTimeSecondForShow(){
		DateFormat whippleTimeFormat = new SimpleDateFormat(TEXT_DATE_TIME);
		return whippleTimeFormat.format(new Date(System.currentTimeMillis()));
	}

	/**
     * 说明：返回指定时间之后的时间，可正可负，例如：给定20071201000000，减10天，得到新的日期
     * 作者：时克英
     * 时间：12 7, 2007
     */
    public static long getAfterLongTime(long oldTime, int dayNum){
    	Calendar calendar = Calendar.getInstance();
    	String strDate = String.valueOf(oldTime);
    	int year = Integer.parseInt(strDate.substring(0, 4));
        int month = Integer.parseInt(strDate.substring(4, 6)) - 1;
        int day = Integer.parseInt(strDate.substring(6, 8));
        int hour = Integer.parseInt(strDate.substring(8, 10));
        int minute = Integer.parseInt(strDate.substring(10,12));
        int second = Integer.parseInt(strDate.substring(12, 14));

        calendar.set(year, month, day, hour, minute, second);
        calendar.add(Calendar.DAY_OF_MONTH, dayNum);

		DateFormat whippleTimeFormat = new SimpleDateFormat(TEXT_DATE_TIME);
        String result = whippleTimeFormat.format(calendar.getTime());
        return Long.parseLong(result);
    }

    /**
     * 返回指定天数之后的日期时间，如：2018-03-29 13:57:01
     * @param daysNumber 增加的天数，可为负值表示之前几天
     * @return
     */
    public static String getShowDateTimeAfterDays(int daysNumber) {
        Calendar now = Calendar.getInstance();
        now.add(Calendar.DATE, daysNumber);
		DateFormat whippletreeTimeFormat = new SimpleDateFormat(TEXT_DATE_TIME_WITH_LINE);
        return whippletreeTimeFormat.format(now.getTime());
    }

    /**
     * 返回当前年月日，可以增减天数
     * @param afterDay 增加的天数（可为负值表示递减）
     * @return
     */
    public static int[] getYearMonthDayArray(int afterDay){
    	Calendar now = Calendar.getInstance();
        now.add(Calendar.DATE, afterDay);
        int[] yearMonthDay = new int[3];
        yearMonthDay[0] = now.get(Calendar.YEAR);
        yearMonthDay[1] = now.get(Calendar.MONTH) + 1;
        yearMonthDay[2] = now.get(Calendar.DAY_OF_MONTH);
        return yearMonthDay;
    }

    /**
     * 返回指定天数之后的日期时间，如：2018-03-29
     * @param daysNumber 增加的天数，可为负值表示之前几天
     * @return
     */
    public static String getShowDateAfterDays(int daysNumber) {
        Calendar now = Calendar.getInstance();
        now.add(Calendar.DATE, daysNumber);
		DateFormat whippletreeDateFormat = new SimpleDateFormat(TEXT_DATE_WITH_LINE);
        return whippletreeDateFormat.format(now.getTime());
    }

    /**
     * 返回给定日期的后续几天的日期值
     * @param year 当前年份，如：2019
     * @param month 当前月份，如：1月
     * @param dayOfMonth 当前日期，如：2号
     * @param afterDayNum 今后几天
     * @return 返回数组，3个值，如：[2019,1,2]
     */
    public static int[] getNextDay(int year, int month, int dayOfMonth, int afterDayNum){
    	LocalDate localDate = LocalDate.of(year, month, dayOfMonth);
    	localDate = localDate.minusDays(-afterDayNum);
    	return new int[]{localDate.getYear(), localDate.getMonthValue(), localDate.getDayOfMonth()};
    }

    /**
     * 返回当前日期是星期几
     * @return 1~7
     */
    public static int getCurrentWeekdayValue(){
        LocalDate currentDate = LocalDate.now();
        return currentDate.getDayOfWeek().getValue();
    }

    /**
     * 返回当前年份
     * @return
     */
    public static String getCurrentYear(){
    	return getDateForHuman(System.currentTimeMillis()).substring(0, 4);
    }

    public static int getCurrentYearInteger(){
    	Calendar now = Calendar.getInstance();
    	return now.get(Calendar.YEAR);
    }
    /**
     * 返回当前年月信息，格式：2020-12
     * @return
     */
    public static String getCurrentYearMonthForHuman(){
    	String showDateOrTime = getDateForHuman(System.currentTimeMillis());
    	return showDateOrTime.substring(0, 7);
    }

    public static int getCurrentHour(){
    	Calendar calendar = Calendar.getInstance();
    	return calendar.get(Calendar.HOUR_OF_DAY);
    }

    public static int getCurrentMinute(){
    	Calendar calendar = Calendar.getInstance();
    	return calendar.get(Calendar.MINUTE);
    }

    /**
     * 返回当前小时：分钟
     * @param afterMinutes 可以增加分钟（正负都可）
     * @return
     */
    public static int[] getCurrentHourMinute(int afterMinutes){
    	Calendar calendar = Calendar.getInstance();
    	calendar.add(Calendar.MINUTE, afterMinutes);
    	int[] hourMinute = new int[2];
    	hourMinute[0] = calendar.get(Calendar.HOUR_OF_DAY);
    	hourMinute[1] = calendar.get(Calendar.MINUTE);
    	return hourMinute;
    }

    /**
     * 返回当前年月日，放在数组中
     * @return
     */
    public static int[] getCurrentYearMonthDay(){
    	int[] result = new int[3];
    	Calendar now = Calendar.getInstance();
    	result[0] = now.get(Calendar.YEAR);
    	result[1] = now.get(Calendar.MONTH) + 1;
    	result[2] = now.get(Calendar.DAY_OF_MONTH);
    	return result;
    }

    public static int[] getCurrentYearMonthDayHourMinute(){
    	int[] result = new int[5];
    	Calendar now = Calendar.getInstance();
    	result[0] = now.get(Calendar.YEAR);
    	result[1] = now.get(Calendar.MONTH) + 1;
    	result[2] = now.get(Calendar.DAY_OF_MONTH);
    	result[3] = now.get(Calendar.HOUR_OF_DAY);
    	result[4] = now.get(Calendar.MINUTE);
    	return result;
    }

    public static int[] getCurrentYearMonthDayHourMinute(int monthAdd, int dayAdd, int hourAdd, int minuteAdd){
    	int[] result = new int[5];
    	Calendar now = Calendar.getInstance();
    	if(monthAdd != 0){
    		now.add(Calendar.MONTH, monthAdd);
    	}
    	if(dayAdd != 0){
    		now.add(Calendar.DATE, dayAdd);
    	}
    	if(hourAdd != 0){
    		now.add(Calendar.HOUR, hourAdd);
    	}
    	if(minuteAdd != 0){
    		now.add(Calendar.MINUTE, minuteAdd);
    	}
    	result[0] = now.get(Calendar.YEAR);
    	result[1] = now.get(Calendar.MONTH) + 1;
    	result[2] = now.get(Calendar.DAY_OF_MONTH);
    	result[3] = now.get(Calendar.HOUR_OF_DAY);
    	result[4] = now.get(Calendar.MINUTE);
    	return result;
    }

    /**
     * 提供一个增加的秒数，计算之后的时间。
     * @param addSeconds 要增加的秒数
     * @return
     * @date 2021-10-25
     */
    public static int[] getCurrentYearMonthDayHourMinuteSecond(int addSeconds){
    	int[] result = new int[6];
    	Calendar now = Calendar.getInstance();
    	now.add(Calendar.SECOND, addSeconds);
    	result[0] = now.get(Calendar.YEAR);
    	result[1] = now.get(Calendar.MONTH) + 1;
    	result[2] = now.get(Calendar.DAY_OF_MONTH);
    	result[3] = now.get(Calendar.HOUR_OF_DAY);
    	result[4] = now.get(Calendar.MINUTE);
    	result[5] = now.get(Calendar.SECOND);
    	return result;
    }

    public static int[] getCurrentYearMonthDay(int afterDays){
    	int[] result = new int[3];
    	Calendar now = Calendar.getInstance();
    	now.add(Calendar.DAY_OF_MONTH, afterDays);
    	result[0] = now.get(Calendar.YEAR);
    	result[1] = now.get(Calendar.MONTH) + 1;
    	result[2] = now.get(Calendar.DAY_OF_MONTH);
    	return result;
    }

	/**
	 * @param args
	 */
	public static void main(String[] args) throws Exception {
//		System.out.println("+++++++++++ " + toIntegerDate("2014-08-29"));
		String showDate = "2013-11-20 11:18:21";
//		String showDate = "2013-11-20";
		System.out.println("normal:" + getDateLong(showDate));
//		long e = getDateLongEarly(showDate);
//		long l = getDateLongLate(showDate);
//		System.out.println("early : " + e);
//		System.out.println("late  : " + l);
//		System.out.println("相差      : " + (l - e)/(1000 * 60 * 60));
//		System.out.println("++++++ " + getDateTimeForHuman(new Date().getTime()));
//		System.out.println("今天最早: " + getTodayLongEarly());
//		System.out.println("今天最晚: " + getTodayLongLate());
//		System.out.println(getCurrentMonthFirstDayEarly());
//		System.out.println(getDateTimeSecondForShow());
//		System.out.println(getDateNumber(System.currentTimeMillis()));
//		System.out.println(getAfterLongTime(20170925141302L, 1));
//		System.out.println(getShowDateTimeAfterDays(-7));
//		System.out.println(DateUtils.toLongDateTime("2018-12-05"));

//		System.out.println(getNextDay(2018, 12, 31, 2));
//		System.out.println(getCurrentWeekdayValue());
//		System.out.println(getShowDateAfterDays(0));
//		System.out.println(getCurrentYearMonthForHuman());
//		System.out.println(getCurrentYear());
//		System.out.println(getCurrentHour());
//		System.out.println(toLongDateTime("2019-05-20 02:30:00"));
//		int[] yearMonthDay = getCurrentYearMonthDay();
//		System.out.println(yearMonthDay[0] + ", " + yearMonthDay[1] + ", " + yearMonthDay[2]);
//		int[] ymd = getYearMonthDayArray(-1230);
//		System.out.println(ymd[0] + ", " + ymd[1] + ", " + ymd[2]);
//		int[] hm = getCurrentHourMinute(-100);
//		System.out.println(hm[0] + ", " + hm[1]);

//		int[] ymd = getCurrentYearMonthDay(-3);
//		System.out.println(ymd[0] + ", " + ymd[1] + ", " + ymd[2]);

//		System.out.println(toMillSeconds(20140829102501L));

//		int[] result = getCurrentYearMonthDayHourMinute(1, 0, 1, 10);
//		System.out.println(result[0]);
//		System.out.println(result[1]);
//		System.out.println(result[2]);
//		System.out.println(result[3]);
//		System.out.println(result[4]);

//		int[] result = getCurrentYearMonthDayHourMinuteSecond(30);
//		System.out.println(result[0]);
//		System.out.println(result[1]);
//		System.out.println(result[2]);
//		System.out.println(result[3]);
//		System.out.println(result[4]);
//		System.out.println(result[5]);
		System.out.println(getDateForHuman(System.currentTimeMillis()));

		int [] secondDay = getNextDay(2023, 12, 31, 1);
		System.out.println(secondDay[0] + "," + secondDay[1] + "," + secondDay[2]);
	}

}
