package com.walker.infrastructure.utils;

import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

import java.util.List;
import java.util.Map;

public class JsonUtils {

    public static final ObjectMapper objectMapper = new ObjectMapper();

    /**
     * 把JSON字符串转换成集合对象。
     * @param json 原始json字符串
     * @param clazz 集合中的泛型类型
     * @return
     * @param <T>
     * @throws Exception
     * @date 2023-06-26
     */
    public static final <T> List<T> jsonStringToList(String json, Class<T> clazz) throws Exception{
        JavaType javaType = objectMapper.getTypeFactory().constructParametricType(List.class, clazz);
        return objectMapper.readValue(json, javaType);
    }

    /**
     * 把java对象转成Json格式的'ObjectNode'。
     * @param javaObj
     * @return
     * @date 2023-03-24
     */
    public static final ObjectNode javaObjectToObjectNode(Object javaObj){
        return objectMapper.convertValue(javaObj, ObjectNode.class);
    }

    /**
     * 把java对象转换为JSON字符串
     * @param javaObject
     * @return
     * @throws RuntimeException
     */
    public static String objectToJsonString(Object javaObject) throws Exception{
        return objectMapper.writeValueAsString(javaObject);
//        try {
//        } catch (JsonProcessingException e) {
//            throw new RuntimeException("objectToJsonString,对象转json字符串异常：" + e.getMessage(), e);
//        }
    }

    public static byte[] objectToBytes(Object javaObject) throws Exception{
        if(javaObject == null){
            return null;
        }
        return objectMapper.writeValueAsBytes(javaObject);
    }

    public static <T> T jsonStringToObject(String json, Class<T> clazz) throws Exception{
        return objectMapper.readValue(json, clazz);
    }

    public static ObjectNode jsonStringToObjectNode(String json) throws Exception{
        return jsonStringToObject(json, ObjectNode.class);
    }

    public static ObjectNode mapToObjectNode(Map<String, Object> map){
        ObjectNode node = objectMapper.createObjectNode();
        if(map != null){
            for(Map.Entry<String, Object> entry : map.entrySet()){
                node.put(entry.getKey(), entry.getValue().toString());
            }
        }
        return node;
    }

    /**
     * 把Json集合对象转换成json array 字符串。<br/>
     * 该方法在Jdbc代码生成模块中使用。
     * @param objectNodeList
     * @return
     * @throws Exception
     * @date 2022-08-17
     */
    public static String toJsonArray(List<ObjectNode> objectNodeList) throws Exception{
        ArrayNode arrayNode = objectMapper.createArrayNode();
        if(!StringUtils.isEmptyList(objectNodeList)){
            for(ObjectNode on : objectNodeList){
                arrayNode.add(on);
            }
        }
        return objectMapper.writeValueAsString(arrayNode);
    }

    public static ArrayNode toJsonArray(String jsonArray) throws Exception{
        return objectMapper.readValue(jsonArray, ArrayNode.class);
    }

    /**
     * 判断<code>JsonNode</code> 是否空对象，因为存在 'null' 字符串的情况。
     * @param jsonNode
     * @return
     * @date 2022-11-15
     */
    public static final boolean isEmptyObject(JsonNode jsonNode){
        if(jsonNode == null){
            return true;
        }
        if(jsonNode.toString().equals(StringUtils.NULL_STRING)){
            return true;
        }
        return false;
    }

}
