package com.walker.push;

import com.walker.infrastructure.core.ApplicationBeanInitialized;

import java.util.List;

/**
 * 推送者定义。
 * <p>1.每个推送者使用一种推送通道，实现一种类型推送</p>
 * <pre>
 *     T 提供了推送内部转换对象，把业务转换成通道可识别的内部对象发送，如：短信（SmsMessage）等。
 * </pre>
 * @author 时克英
 * @date 2023-04-21
 */
public interface Pushable<T> extends ApplicationBeanInitialized {

    /**
     * 返回推送者ID，唯一
     * @return
     */
    String getId();

    /**
     * 返回推送者名称
     * @return
     */
    String getName();

    /**
     * 返回推送者支持的通道，只能一个。
     * @return
     */
    NotificationChannel getNotificationChannel();

    /**
     * 推送方法。
     * @param notification
     * @return
     * @throws PushException
     */
    PushResult push(Notification notification) throws PushException;

    /**
     * 把通知对象转换成目标可用的数据对象
     * @param notification
     * @return 一个通知可能包含多个接收者，所以返回是个业务对象集合。
     * @date 2023-04-22
     */
    List<T> translateToTarget(Notification notification);

    /**
     * 推送者是否支持异步执行，如果支持则在保存时需要重写回调方法。
     * @return
     * @date 2023-04-24
     */
    boolean supportAsync();

    /**
     * 在异步任务中，设置回调机制，当推送者是异步调用时必须传递该对象。
     * @param listener
     * @date 2023-04-24
     */
    void setPushStatusListener(PushStatusListener listener);

    /**
     * 返回异步推送的监听器对象，如果同步则为空。
     * @return
     * @date 2023-04-27
     */
    PushStatusListener getPushStatusListener();
}
