package com.walker.jdbc.sqlgen;

import com.walker.jdbc.SqlAndParameters;
import com.walker.jdbc.dao.PersistenceException;
import org.apache.commons.lang3.ArrayUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Insert语句生成工具
 *
 * @author 时克英
 * @date 2017年3月3日
 */
public class InsertBuilder extends AbstractSqlBuilder {
    private final String table;
    private final List<String> columns = new ArrayList<>();
    private final List<Object> values = new ArrayList<>();

    /**
     * 根据表名创建insert生成工具
     *
     * @param table 表名
     */
    public InsertBuilder(String table) {
        this.table = table;
    }

    /**
     * 设置插入语句的字段和值
     *
     * @param column 字段名称
     * @param value  字段值
     * @return InsertBuilder 对象
     */
    public InsertBuilder set(String column, Object value) {
        this.columns.add(column);
        this.values.add(value);
        return this;
    }

    /**
     * 设置插入语句的字段和值
     *
     * @param column 字段名称
     * @param value  字段值
     * @param isSet  是否设置了值
     * @return InsertBuilder 对象
     */
    public InsertBuilder set(String column, Object value, boolean isSet) {
        if (isSet) {
            this.columns.add(column);
            this.values.add(value);
        }
        return this;
    }

    /**
     * 设置插入语句的字段和值
     *
     * @param map 插入语句需要的字段和值得集合
     * @return InsertBuilder 对象
     */
    public InsertBuilder set(Map<String, Object> map) {
        for (Map.Entry<String, Object> entry : map.entrySet()) {
            this.set(entry.getKey(), entry.getValue());
        }
        return this;
    }

    /**
     * @see AbstractSqlBuilder#genMapSql()
     */
    @Override
    public SqlAndParameters<Map<String, Object>> genMapSql() {
        Map<String, Object> parametersMap = new HashMap<>();

        StringBuilder sb = new StringBuilder();
        sb.append("insert into ").append(this.table).append(" (");

        for (int i = 0; i < this.columns.size(); i++) {
            if (i > 0) {
                sb.append(", ");
            }
            sb.append(this.columns.get(i));
        }
        sb.append(") values (");
        for (int i = 0; i < this.columns.size(); i++) {
            if (i > 0) {
                sb.append(", ");
            }
            sb.append(":").append(this.columns.get(i));
            parametersMap.put(this.columns.get(i), getNotEmptyValue(this.values.get(i)));
        }
        sb.append(")");

        return new SqlAndParameters<>(sb.toString(), parametersMap);
    }

    /**
     * 此方法不支持
     *
     * @param where         条件片段
     * @param parametersMap 条件参数
     * @return SqlAndParameters
     */
    @Override
    public SqlAndParameters<Map<String, Object>> genMapSql(String where, Map parametersMap) {
        throw new PersistenceException("不支持的方法！");
    }

    /**
     * @see AbstractSqlBuilder#genArraySql()
     */
    @Override
    public SqlAndParameters<Object[]> genArraySql() {
        Object[] parametersArray = new Object[]{};

        StringBuilder sb = new StringBuilder();
        sb.append("insert into ").append(this.table).append(" (");

        for (int i = 0; i < this.columns.size(); i++) {
            if (i > 0) {
                sb.append(", ");
            }
            sb.append(this.columns.get(i));
        }
        sb.append(") values (");
        for (int i = 0; i < this.columns.size(); i++) {
            if (i > 0) {
                sb.append(", ");
            }
            sb.append("?");
            parametersArray = ArrayUtils.add(parametersArray, getNotEmptyValue(this.values.get(i)));
        }
        sb.append(")");

        return new SqlAndParameters<>(sb.toString(), parametersArray);
    }

    /**
     * 此方法不支持
     *
     * @param where           条件片段
     * @param parametersArray 条件参数
     * @return SqlAndParameters
     */
    @Override
    public SqlAndParameters<Object[]> genArraySql(String where, Object[] parametersArray) {
        throw new PersistenceException("不支持的方法！");
    }
}
