package com.walker.jdbc.sqlgen;

import com.walker.jdbc.SqlAndParameters;
import org.apache.commons.lang3.ArrayUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * select 语句生成工具
 *
 * @author 时克英
 * @date 2017年3月3日
 */
public class SelectBuilder extends AbstractSqlBuilder {
    private final String table;

    private final List<String> columns = new ArrayList<>();
    private final List<Object> values = new ArrayList<>();

    /**
     * 更近表名创建sql查询对象
     *
     * @param table 表名
     */
    public SelectBuilder(String table) {
        this.table = table;
    }

    /**
     * 设置where条件中一个字段的值，所有的字段之间以and方式连接
     *
     * @param column 字段名
     * @param value  字段值
     * @return SelectBuilder对象
     */
    public SelectBuilder where(String column, Object value) {
        this.columns.add(column);
        this.values.add(value);
        return this;
    }

    /**
     * @see AbstractSqlBuilder#genMapSql()
     */
    public SqlAndParameters<Map<String, Object>> genMapSql() {
        StringBuilder sb = new StringBuilder("select * from ").append(this.table).append(" where ");
        Map<String, Object> parametersMap = new HashMap<>();
        for (int i = 0; i < this.columns.size(); i++) {
            if (i > 0) {
                sb.append(" and ");
            }
            if (isSqlNull(this.values.get(i))) {
                sb.append(this.columns.get(i)).append(" is null");
            } else {
                sb.append(this.columns.get(i)).append("=").append(":").append(this.columns.get(i));
                parametersMap.put(this.columns.get(i), this.values.get(i));
            }
        }
        return new SqlAndParameters<>(sb.toString(), parametersMap);
    }


    /**
     * @see AbstractSqlBuilder#genArraySql()
     */
    public SqlAndParameters<Object[]> genArraySql() {
        StringBuilder sb = new StringBuilder("select * from ").append(this.table).append(" where ");
        Object[] parametersArray = new Object[]{};
        for (int i = 0; i < this.columns.size(); i++) {
            if (i > 0) {
                sb.append(" and ");
            }
            if (isSqlNull(this.values.get(i))) {
                sb.append(this.columns.get(i)).append(" is null");
            } else {
                sb.append(this.columns.get(i)).append("=?");
                parametersArray = ArrayUtils.add(parametersArray, this.values.get(i));
            }
        }
        return new SqlAndParameters<>(sb.toString(), parametersArray);
    }

    /**
     * @see AbstractSqlBuilder#genMapSql(String, Map< String, Object> )
     */
    @Override
    public SqlAndParameters<Map<String, Object>> genMapSql(String where, Map<String, Object> parametersMap) {
        return new SqlAndParameters<>("select * from " + this.table + " " + where, parametersMap);
    }

    /**
     * @see AbstractSqlBuilder#genArraySql(String, Object[])
     */
    @Override
    public SqlAndParameters<Object[]> genArraySql(String where, Object[] parametersArray) {
        return new SqlAndParameters<>("select * from " + this.table + " " + where, parametersArray);
    }
}