package com.walker.jdbc;

import com.walker.jdbc.util.StringSqlUtils;
import com.walker.jdbc.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.Serializable;
import java.lang.reflect.Array;
import java.util.Date;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * sql语句及参数
 *
 * @param <T> 参数类型
 * @author 时克英
 * @date 2022-08-12
 */
public class SqlAndParameters<T> implements Serializable {
    // 序列化版本号
//    private static final long serialVersionUID = 1L;
    public static final long serialVersionUID = 123456787654321L;

    private final String sql;
    private final T parameters;
    private static final Pattern MAP_PARAMETER_PLACEHOLDER = Pattern.compile(":([a-zA-Z_][a-zA-Z0-9_\\.]*)");
    private static final Pattern ARRAY_PARAMETER_PLACEHOLDER = Pattern.compile("\\?");

    private static final Logger logger = LoggerFactory.getLogger(SqlAndParameters.class);

    /**
     * 通过sql和参数构造对象
     *
     * @param sql        查询语句
     * @param parameters 参数对象
     */
    public SqlAndParameters(String sql, T parameters) {
        this.sql = sql;
        this.parameters = parameters;
    }

    public String getSql() {
        return sql;
    }

    public T getParameters() {
        return parameters;
    }

    @Override
    public String toString() {
//        return ToStringBuilder.reflectionToString(this, ToStringStyle.MULTI_LINE_STYLE);
        return new StringBuilder(this.sql).append("\r\n")
                .append(this.parameters).toString();
    }

    /**
     * 获取替换参数后的SQL语句
     *
     * @return
     */
    public String toPrintSql() {
        if (this.parameters == null) {
            return sql;
        }
        try {
            if (this.parameters instanceof Map) {
                Matcher matcher = MAP_PARAMETER_PLACEHOLDER.matcher(this.sql);
                StringBuffer sb = new StringBuffer();
                while (matcher.find()) {
                    if (((Map) parameters).containsKey(matcher.group(1))) {
                        String value = getSqlParamString(((Map) parameters).get(matcher.group(1)));
                        matcher.appendReplacement(sb, Matcher.quoteReplacement(value));
                    } else {
                        matcher.appendReplacement(sb, matcher.group());
                    }
                }
                matcher.appendTail(sb);
                return sb.toString();
            } else if (this.parameters.getClass().isArray()) {
                Matcher matcher = ARRAY_PARAMETER_PLACEHOLDER.matcher(this.sql);
                StringBuffer sb = new StringBuffer();
                int i = 0;
                while (matcher.find()) {
                    String value = getSqlParamString(Array.get(this.parameters, i++));
                    matcher.appendReplacement(sb, Matcher.quoteReplacement(value));
                }
                matcher.appendTail(sb);
                return sb.toString();
            } else {
                return sql;
            }
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            return this.toString();
        }
    }

    /**
     * 获取sql参数
     *
     * @param value
     * @return
     */
    private String getSqlParamString(Object value) {
        if (value == null || StringUtils.EMPTY_STRING.equals(value)) {
            return StringUtils.NULL_STRING;
        }
        if (value instanceof String || value instanceof Date) {
            return new StringBuilder(StringUtils.SINGLE_QUOTATION)
                    .append(StringSqlUtils.escapeSql(value.toString()))
                    .append(StringUtils.SINGLE_QUOTATION).toString();
//            return StringUtils.SINGLE_QUOTATION + StringSqlUtils.escapeSql(value.toString()) + "'";
        } else {
            return value.toString();
        }
    }
}
