package com.walker.jdbc.sqlgen;

import com.walker.jdbc.SqlAndParameters;
import org.apache.commons.lang3.ArrayUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * update 语句生成工具
 *
 * @author 时克英
 * @date 2017-3-32
 */
public class UpdateBuilder extends AbstractSqlBuilder {
    private final String table;
    private final List<String> setColumns = new ArrayList<>();
    private final List<Object> setValues = new ArrayList<>();

    private final List<String> whereColumns = new ArrayList<>();
    private final List<Object> whereValues = new ArrayList<>();

    /**
     * 根据表名构造更新语句生成工具
     *
     * @param table 表名
     */
    public UpdateBuilder(String table) {
        this.table = table;
    }

    /**
     * 设置update语句更新的字段值
     *
     * @param column 字段名
     * @param value  字段值
     * @return UpdateBuilder对象
     */
    public UpdateBuilder set(String column, Object value) {
        return this.set(column, value, true);
    }

    /**
     * 设置update语句更新的字段值
     *
     * @param column 字段名
     * @param value  字段值
     * @param enable 是否更新
     * @return UpdateBuilder对象
     */
    public UpdateBuilder set(String column, Object value, boolean enable) {
        if (enable) {
            this.setColumns.add(column);
            this.setValues.add(value);
        }
        return this;
    }

    /**
     * 设置更新时的where条件字段
     *
     * @param column 条件字段名
     * @param value  字段值
     * @return UpdateBuilder对象
     */
    public UpdateBuilder where(String column, Object value) {
        this.whereColumns.add(column);
        this.whereValues.add(value);
        return this;
    }


    /**
     * @see AbstractSqlBuilder#genMapSql()
     */
    @Override
    public SqlAndParameters<Map<String, Object>> genMapSql() {
        Map<String, Object> parametersMap = new HashMap<>();

        StringBuilder sb = new StringBuilder("update ").append(this.table);
        sb.append(" set ");
        for (int i = 0; i < this.setColumns.size(); i++) {
            if (i > 0) {
                sb.append(", ");
            }
            sb.append(this.setColumns.get(i)).append("=:").append(this.setColumns.get(i) + "_");
            parametersMap.put(this.setColumns.get(i) + "_", getNotEmptyValue(this.setValues.get(i)));
        }
        sb.append(" where ");
        for (int i = 0; i < this.whereColumns.size(); i++) {
            if (i > 0) {
                sb.append(" and ");
            }
            if (isSqlNull(whereValues.get(i))) {
                sb.append(this.whereColumns.get(i)).append(" is null");
            } else {
                sb.append(this.whereColumns.get(i)).append("=:").append(this.whereColumns.get(i));
                parametersMap.put(this.whereColumns.get(i), this.whereValues.get(i));
            }
        }
        return new SqlAndParameters<>(sb.toString(), parametersMap);
    }

    /**
     * @see AbstractSqlBuilder#genMapSql(String, Map< String, Object> )
     */
    @Override
    public SqlAndParameters<Map<String, Object>> genMapSql(String where, Map<String, Object> parametersMap) {
        Map<String, Object> temp = new HashMap<>();

        StringBuilder sb = new StringBuilder("update ").append(this.table);
        sb.append(" set ");
        for (int i = 0; i < this.setColumns.size(); i++) {
            if (i > 0) {
                sb.append(", ");
            }
            sb.append(this.setColumns.get(i)).append("=:").append(this.setColumns.get(i) + "_");
            temp.put(this.setColumns.get(i) + "_", getNotEmptyValue(this.setValues.get(i)));
        }
        sb.append(" ").append(where);

        temp.putAll(parametersMap);

        return new SqlAndParameters<>(sb.toString(), temp);
    }

    /**
     * @see AbstractSqlBuilder#genArraySql(String, Object[])
     */
    @Override
    public SqlAndParameters<Object[]> genArraySql(String where, Object[] parametersArray) {
        Object[] temp = new Object[]{};

        StringBuilder sb = new StringBuilder("update ").append(this.table);
        sb.append(" set ");
        for (int i = 0; i < this.setColumns.size(); i++) {
            if (i > 0) {
                sb.append(", ");
            }
            sb.append(this.setColumns.get(i)).append("=?");
            temp = ArrayUtils.add(temp, getNotEmptyValue(this.setValues.get(i)));
        }
        sb.append(" ").append(where);
        temp = ArrayUtils.addAll(temp, parametersArray);

        return new SqlAndParameters<>(sb.toString(), temp);
    }

    /**
     * @see AbstractSqlBuilder#genArraySql()
     */
    @Override
    public SqlAndParameters<Object[]> genArraySql() {
        Object[] parametersArray = new Object[]{};

        StringBuilder sb = new StringBuilder("update ").append(this.table);
        sb.append(" set ");
        for (int i = 0; i < this.setColumns.size(); i++) {
            if (i > 0) {
                sb.append(", ");
            }
            sb.append(this.setColumns.get(i)).append("=?");
            parametersArray = ArrayUtils.add(parametersArray, getNotEmptyValue(this.setValues.get(i)));
        }
        sb.append(" where ");
        for (int i = 0; i < this.whereColumns.size(); i++) {
            if (i > 0) {
                sb.append(" and ");
            }
            if (isSqlNull(whereValues.get(i))) {
                sb.append(this.whereColumns.get(i)).append(" is null");
            } else {
                sb.append(this.whereColumns.get(i)).append("=?");
                parametersArray = ArrayUtils.add(parametersArray, this.whereValues.get(i));
            }
        }
        return new SqlAndParameters<>(sb.toString(), parametersArray);
    }
}
